﻿using System.Collections.Generic;
using System.Drawing;
using System.Drawing.Imaging;
using System.IO;
using System.Reflection;

namespace AStar
{
    class MapRenderer
    {
        // load a bitmap, by name, from the Images resources
        static Bitmap LoadBitmap(string name)
        {
            using (Stream file = Assembly.GetExecutingAssembly().GetManifestResourceStream("AStar.Images." + name))
                return new Bitmap(Image.FromStream(file));
        }


        // set the alpha channel to transparent of pixels of a bitmap that match the specified color
        // see http://stackoverflow.com/questions/1514846/how-to-load-pngs-and-export-to-tga-keeping-alpha-in-c 
        static Bitmap MakeTransparent(Bitmap bmp, byte r, byte g, byte b)
        {
            BitmapData data = bmp.LockBits(new Rectangle(0, 0, bmp.Width, bmp.Height),
                ImageLockMode.ReadOnly, bmp.PixelFormat);
            unsafe
            {
                for (int x = 0; x < data.Width; x++)
                {
                    int i = x * 4;
                    for (int y = 0; y < data.Height; y++)
                    {
                        byte* row = (byte*)data.Scan0 + (y * data.Stride);
                        // BGRA order
                        if ((row[i] == b) && (row[i + 1] == g) && (row[i + 2] == r))
                            row[i + 3] = 0;  // transparent
                    }
                }
            }
            bmp.UnlockBits(data);
            return bmp;
        }


        static string Neighbor(string[] map, char dir, int y, int x)
        {
            char current = map[y][x];
            if (dir == 'N') y = y - 1;
            if (dir == 'S') y = y + 1;
            if (dir == 'W') x = x - 1;
            if (dir == 'E') x = x + 1;
            if ((y >= 0) && (y < map.Length) && (x >= 0) && (x < map[y].Length))
                if (map[y][x] == current)
                    return dir.ToString();
            return string.Empty;
        }


        static Bitmap LoadTile(string[] map, int y, int x)
        {
            // TILES WERE RENAMED to fit this pattern consistently
            string name = string.Empty;
            char current = map[y][x];
            string suffix =
                Neighbor(map, 'N', y, x) +
                Neighbor(map, 'E', y, x) +
                Neighbor(map, 'S', y, x) +
                Neighbor(map, 'W', y, x);

            if (current == 'G')
                name = "L1_Terrain001";

            if (current == 'S')
                name = "L1_Terrain035";

            if (current == '1')
                name = "L2_Terrain084";

            if (current == '2')
                name = "L2_Terrain087";

            if (current == 't')
                name = "L2_Terrain098";

            if (current == 'R')
                name = "L2_RoadDirt" + suffix;

            if (current == 'W')
                name = "L2_WallStone" + suffix;

            if (current == 'v')
                name = "L2_MountainBigVolcano01";

            if (current == 'V')
                name = "L2_MountainBigVolcanoErupt01";

            if (!string.IsNullOrEmpty(name))
                return MakeTransparent(LoadBitmap(name + ".PNG"), 255, 0, 255);

            return null;
        }


        // tile size is 54x54, but only bottom half is normally used
        // reduce by 2 pixels for better overlap
        const int tileWidth = 52;
        const int tileHeight = 26;  // full height is 54

        static void MapToXY(int mapHeight, int i, int j, out int x, out int y, out int cx, out int cy)
        {
            x = (j * tileWidth / 2) + (i * tileWidth / 2);
            y = (i * tileHeight / 2) - (j * tileHeight / 2) + ((mapHeight - 1) * tileHeight / 2);
            cx = x + tileWidth / 2;
            cy = y + tileHeight + tileHeight / 2;
        }


        // need to add the path in, too
        static public Bitmap Render(Map map, Pos[] path)
        {
            int mapWidth = map.MapData[0].Length;
            int mapHeight = map.MapData.Length;

            Bitmap bmpMap = new Bitmap(
                tileWidth * mapWidth + tileWidth / 2,
                tileHeight * (mapHeight + 1));
            Graphics gMap = Graphics.FromImage((Image)bmpMap);

            int x, y, cx, cy, px, py, pcx, pcy;

            // based on: http://stackoverflow.com/questions/892811/drawing-isometric-game-worlds
            for (int i = 0; i < mapHeight; i++)
                for (int j = mapWidth - 1; j >= 0; j--)
                {
                    MapToXY(mapHeight, i, j, out x, out y, out cx, out cy);

                    using (Bitmap b = LoadTile(map.MapData, i, j))
                        if (b != null) gMap.DrawImage(b, x, y);

                    using (Bitmap b = LoadTile(map.OverlayData, i, j))
                        if (b != null) gMap.DrawImage(b, x, y);

                }

            
            // draw path last, else a road is drawn on top
            using (var pen = new Pen(Color.Red, 2))
            {
                for (int i = 0; i < path.Length; i++)
                {
                    MapToXY(mapHeight, path[i].Y, path[i].X, out x, out y, out cx, out cy);

                        // draw a circle in the center of the path tile
                        const int width = tileHeight / 3;
                        gMap.FillEllipse(Brushes.Red, cx - width / 2, cy - width / 2, width, width);  // FillEllipse is from the upper left hand corner, not the center

                        // draw a short line in the direction of movement
                        if (i<path.Length-1)
                        {
                            MapToXY(mapHeight, path[i + 1].Y, path[i + 1].X, out px, out py, out pcx, out pcy);

                            // draw a short line toward the next tile in the path
                            gMap.DrawLine(pen, pcx, pcy, pcx + (cx - pcx) / 2, pcy + (cy - pcy) / 2);
                        }
                }
            }
            
            return bmpMap;
        }

    }
}
