using System;
using System.Collections.Generic;
using System.Linq;
using System.IO;
using System.Collections;

namespace AStar
{
    public class Pos
    {
        public int X { get; set; }
        public int Y { get; set; }
        public Pos(int x, int y)
        {
            X = x;
            Y = y;
        }
        public override string ToString()
        {
            return "<" + X + "," + Y + ">";
        }
    }
  
    public class PriorityQueue<P, V>
    {
        private SortedDictionary<P, Queue<V>> list = new SortedDictionary<P, Queue<V>>();
        public void Enqueue(P priority, V value)
        {
            Queue<V> q;
            if (!list.TryGetValue(priority, out q))
            {
                q = new Queue<V>();
                list.Add(priority, q);
            }
            q.Enqueue(value);
        }
        public V Dequeue()
        {
            // will throw if there isnt any first element!
            var pair = list.First();
            var v = pair.Value.Dequeue();
            if (pair.Value.Count == 0) // nothing left of the top priority.
                list.Remove(pair.Key);
            return v;
        }
        public bool IsEmpty
        {
            get { return !list.Any(); }
        }
    }


    public abstract class Map
    {
        protected class Node
        {
            public Map Map { get; private set; }
            public int X { get; private set; }
            public int Y { get; private set; }

            public Node(Map map, int x, int y)
            {
                Map = map;
                X = x;
                Y = y;
                Validate();
            }

            void Validate()
            {
                if ((Y < 0) || (Y >= Map.MapData.Length) ||
                    (X < 0) || (X >= Map.MapData[0].Length))
                    throw new ArgumentOutOfRangeException("The coordinate " + X + "," + Y + " is out of range");
            }


            public override string ToString()
            {
                return "<" + X + "," + Y + ">";
            }


            bool IsNeighbor(int x, int y)
            {
                // not passable if off the map or of certain terrain types, otherwise it is a valid neighbor
                return ((y >= 0) && (y < Map.MapData.Length) &&
                    (x >= 0) && (x < Map.MapData[0].Length)) && Map.IsPassable(x, y);
            }

            public IEnumerable<Node> Neighbours
            {
                get
                {
                    // cardinal directions only
                    var neighbors = new List<Node>();
                    if (IsNeighbor(X, Y - 1)) neighbors.Add(new Node(Map, X, Y - 1));
                    if (IsNeighbor(X, Y + 1)) neighbors.Add(new Node(Map, X, Y + 1));
                    if (IsNeighbor(X - 1, Y)) neighbors.Add(new Node(Map, X - 1, Y));
                    if (IsNeighbor(X + 1, Y)) neighbors.Add(new Node(Map, X + 1, Y));

                    // extended
                    if (IsNeighbor(X - 1, Y - 1)) neighbors.Add(new Node(Map, X - 1, Y - 1));
                    if (IsNeighbor(X - 1, Y + 1)) neighbors.Add(new Node(Map, X - 1, Y + 1));
                    if (IsNeighbor(X - 1, Y + 1)) neighbors.Add(new Node(Map, X - 1, Y + 1));
                    if (IsNeighbor(X + 1, Y + 1)) neighbors.Add(new Node(Map, X + 1, Y + 1));

                    foreach (var n in neighbors)
                        yield return n;
                }
            }

            // need to override these so HashSet.Compare() will work
            public override bool Equals(object obj)
            {
                if (!(obj is Node))
                    return false;
                return Map.Equals(((Node)obj).Map) && X.Equals(((Node)obj).X) && Y.Equals(((Node)obj).Y);
            }

            public override int GetHashCode()
            {
                return Map.GetHashCode() ^ X.GetHashCode() ^ Y.GetHashCode();
            }
        }


        class Path : IEnumerable<Node>
        {
            public Node LastStep { get; private set; }
            public Path PreviousSteps { get; private set; }
            public double TotalCost { get; private set; }
            private Path(Node lastStep, Path previousSteps, double totalCost)
            {
                LastStep = lastStep;
                PreviousSteps = previousSteps;
                TotalCost = totalCost;
            }
            public Path(Node start) : this(start, null, 0) { }
            public Path AddStep(Node step, double stepCost)
            {
                return new Path(step, this, TotalCost + stepCost);
            }
            public IEnumerator<Node> GetEnumerator()
            {
                for (Path p = this; p != null; p = p.PreviousSteps)
                    yield return p.LastStep;
            }
            IEnumerator IEnumerable.GetEnumerator()
            {
                return this.GetEnumerator();
            }
        }


        public string[] MapData { get; private set; }
        public string[] OverlayData { get; private set; }

        public Map(string[] mapData, string[] overlayData)
        {
            MapData = mapData;
            OverlayData = overlayData;
            Validate();
        }

        void Validate()
        {
            if (MapData.Length == 0)
                throw new ArgumentException("Map must contain tiles");
            if (MapData.Length != OverlayData.Length)
                throw new ArgumentException("Map and overlay length doesn't match");
            for (int i = 0; i < MapData.Length; i++)
            {
                if (MapData[i].Length == 0)
                    throw new ArgumentException("Map must contain tiles at row " + i);
                if (MapData[i].Length != OverlayData[i].Length)
                    throw new ArgumentException("Map and overlay length doesn't match at row " + i);
            }
        }

        double EstimatedDistance(Node n1, Node n2)
        {
            // use Euclidean distance to be sure cost is underestimated
            return Math.Sqrt((n2.X - n1.X) * (n2.X - n1.X)) + ((n2.Y - n1.Y) * (n2.Y - n1.Y));
        }

        protected abstract bool IsPassable(int x, int y);

        double NeighborDistance(Node n1, Node n2) {
            // ensure that this really is only called for neighboring nodes
            if ((n1 == n2) || (Math.Abs(n2.X - n1.X) > 1) || (Math.Abs(n2.Y - n1.Y) > 1))
                throw new ArgumentException(n1 + " is not a neighbor of " + n2);

            return Distance(n1, n2);
        }

        protected abstract double Distance(Node n1, Node n2);

        public Pos[] FindPath(Pos startPos, Pos destinationPos)
        {
            var start = new Node(this, startPos.X, startPos.Y);
            var destination = new Node(this, destinationPos.X, destinationPos.Y);
            var closed = new HashSet<Node>();
            var open = new PriorityQueue<double, Path>();
            open.Enqueue(0, new Path(start));
            while (!open.IsEmpty)
            {
                var path = open.Dequeue();
                if (closed.Contains(path.LastStep))
                    continue;
                if (path.LastStep.Equals(destination))
                    return path.Select(node => new Pos(node.X, node.Y)).ToArray();
                closed.Add(path.LastStep);
                foreach (Node n in path.LastStep.Neighbours)
                {
                    double d = NeighborDistance(path.LastStep, n);
                    var newPath = path.AddStep(n, d);
                    open.Enqueue(newPath.TotalCost + EstimatedDistance(n, destination), newPath);
                }
            }
            return null;
        }

    }
}




