using System;
using System.Collections.Generic;
using System.Drawing;  // for Bitmap
using System.Drawing.Imaging;
using System.Linq;

namespace AStar
{
    class StandardMap : Map
    {
        public StandardMap(string[] mapData, string[] overlayData) : base(mapData, overlayData) { }

        protected override bool IsPassable(int x, int y)
        {
            // walls and volcanoes are not passable
            return ((OverlayData[y][x] != 'W') && (OverlayData[y][x] != 'V') && (OverlayData[y][x] != 'v'));
        }

        protected override double Distance(Node n1, Node n2)
        {
            // cost is interpreted as the terrain cost of the square being moved on to
            char m = MapData[n2.Y][n2.X];
            char o = OverlayData[n2.Y][n2.X];

            // Grass=1, Snow=3
            // Road reduces the cost by 1/2
            double distance = 1.0;
            if (m == 'S')
                distance = 3.0;
            if (o == 'R')
                distance /= 2.0;

            // if both X and Y change, then moving in a diagonal, so the distance is greater
            if ((n1.X != n2.X) && (n1.Y != n2.Y))
                distance *= 1.414;

            return distance;
        }
    }

    // VolcanoMap

    class VolcanoMap : StandardMap
    {
        public VolcanoMap(string[] mapData, string[] overlayData) : base(mapData, overlayData) { }

        protected override double Distance(Node n1, Node n2)
        {
            double distance = base.Distance(n1, n2);


            for (int j = 0; j < OverlayData.Length; j++)
                for (int i = 0; i < OverlayData[0].Length; i++)
                    if (OverlayData[j][i] == 'V')
                    {
                        // each erupting volcano influences a radius of 4.5 units
                        // the closer to the volcano, the greater the reduction of path cost (want to get closer)
                        double dV = Math.Sqrt(((i - n2.X) * (i - n2.X)) + ((j - n2.Y) * (j - n2.Y)));
                        if (dV <= 4.5)
                            distance /= ((5.0-dV)*2.0);
                    }

            return distance;
        }
    }

    class VolcanoTombstoneMap : VolcanoMap
    {
        public VolcanoTombstoneMap(string[] mapData, string[] overlayData) : base(mapData, overlayData) { }

        protected override double Distance(Node n1, Node n2)
        {
            double distance = base.Distance(n1, n2);

            for (int j = 0; j < OverlayData.Length; j++)
                for (int i = 0; i < OverlayData[0].Length; i++)
                    if (OverlayData[j][i] == 't')
                    {
                        // each graveyard influences a radius of 2.5 units
                        double dV = Math.Sqrt(((i - n2.X) * (i - n2.X)) + ((j - n2.Y) * (j - n2.Y)));
                        if (dV <= 2.5)
                            distance *= (((3.0 - dV)+0.5)*1.1);
                    }

            return distance;
        }
    }



    class Program
    {
        static void Eval(Map map, Pos start, Pos destination, string name)
        {
            var path = map.FindPath(start, destination);
            Console.WriteLine(name + ": " + string.Join(" ", path.Select(v => v.ToString())));

            using (Bitmap bitmap = MapRenderer.Render(map, path))
                bitmap.Save(name + ".png", ImageFormat.Png);
        }

        static void Main(string[] args)
        {
            Eval(new StandardMap(
                new string[] {
                    "GGGGGGGGGG",
                    "GGGGGGGGGG",
                    "GGGGGGGGGG",
                    "GGGGGGGGGG",
                    "GGGGGGGGGG",
                    "GGGGGGGGGG",
                    "GGGGGGGGGG",
                    "GGGGGGGGGG",
                    "GGGGGGGGGG" ,
                    "GGGGGGGGGG"
                },
                new string[] {
                    "          ",
                    "     R    ",
                    "   RRRRR  ",
                    "     R W  ",
                    "     R W  ",
                    " 1   R W 2",
                    "       W  ",
                    "   WWWWW  ",
                    "       W  ",
                    "          "
                }
            ), new Pos(1, 5), new Pos(9, 5), "roadWallsA");


            Eval(new StandardMap(
                new string[] {
                    "GGGGGGGGGG",
                    "GGGGGGGGGG",
                    "GGGGGGGGGG",
                    "GGGGGGGGGG",
                    "GGGGGGGGGG",
                    "GGGGGGGGGG",
                    "GGGGGGGGGG",
                    "GGGGGGGGGG",
                    "GGGGGGGGGG" ,
                    "GGGGGGGGGG"
                },
                // two more wall segments are enough to make the path take the southern route
                new string[] {
                "          ",
                "     R W  ",  
                "   RRRRW  ",
                "     R W  ",
                "     R W  ",
                " 1   R W 2",
                "       W  ",
                "   WWWWW  ",
                "       W  ",
                "          "
                }
            ), new Pos(1, 5), new Pos(9, 5), "roadWallsB");


            Eval(new VolcanoMap(
                new string[] {
                    "SSSSSSSSSS",
                    "SSSSSSSSSS",
                    "SSSSSSSSSS",
                    "SSSSSSSSSS",
                    "SSSSSSSSSS",
                    "SSSSSSSSSS",
                    "SSSSSSSSSS",
                    "SSSSSSSSSS",
                    "SSSSSSSSSS" ,
                    "SSSSSSSSSS"
                },
                new string[] {
                    "          ",
                    "    v     ",
                    "          ",
                    "          ",
                    "          ",
                    "          ",
                    "1        2",
                    "          ",
                    "          ",
                    "          "
                }
            ), new Pos(0, 6), new Pos(9, 6), "volcanoDormant");


            Eval(new VolcanoMap(
                new string[] {
                    "SSSSSSSSSS",
                    "SSSSSSSSSS",
                    "SSSSSSSSSS",
                    "SSSSSSSSSS",
                    "SSSSSSSSSS",
                    "SSSSSSSSSS",
                    "SSSSSSSSSS",
                    "SSSSSSSSSS",
                    "SSSSSSSSSS" ,
                    "SSSSSSSSSS"
                },
                new string[] {
                    "          ",
                    "          ",
                    "          ",
                    "          ",
                    "    V     ",
                    "          ",
                    "1        2",
                    "          ",
                    "          ",
                    "          "
                }
            ), new Pos(0, 6), new Pos(9, 6), "volcanoErupting");



            Eval(new VolcanoTombstoneMap(
                new string[] {
                    "SSSSSSSSSS",
                    "SSSSSSSSSS",
                    "SSSSSSSSSS",
                    "SSSSSSSSSS",
                    "SSSSSSSSSS",
                    "SSSSSSSSSS",
                    "SSSSSSSSSS",
                    "SSSSSSSSSS",
                    "SSSSSSSSSS" ,
                    "SSSSSSSSSS"
                },
                new string[] {
                    "          ",
                    "    v     ",
                    "          ",
                    "          ",
                    "          ",
                    "   t      ",
                    "1        2",
                    "          ",
                    "          ",
                    "          "
                }
            ), new Pos(0, 6), new Pos(9, 6), "volcanoDormantTombstone");


            Eval(new VolcanoTombstoneMap(
                new string[] {
                    "SSSSSSSSSS",
                    "SSSSSSSSSS",
                    "SSSSSSSSSS",
                    "SSSSSSSSSS",
                    "SSSSSSSSSS",
                    "SSSSSSSSSS",
                    "SSSSSSSSSS",
                    "SSSSSSSSSS",
                    "SSSSSSSSSS" ,
                    "SSSSSSSSSS"
                },
                new string[] {
                    "          ",
                    "    V     ",
                    "          ",
                    "          ",
                    "          ",
                    "   t      ",
                    "1        2",
                    "          ",
                    "          ",
                    "          "
                }
            ), new Pos(0, 6), new Pos(9, 6), "volcanoEruptingTombstone");

        }
    }
}
