#include "DataServerConnectorLib.h"
#include <iostream>
#include <vcclr.h>
#include <sstream>
#include <exception>
#include <string>
#include "Shared/LNK4248.h"
#include <orbsvcs/CosNamingC.h>

// DDS
#include "DatabaseNotification/DatabaseNotificationTypeSupportImpl.h"
#include <dds/DCPS/Service_Participant.h>
#include <dds/DCPS/Marked_Default_Qos.h>
#include <dds/DCPS/SubscriberImpl.h>
#include <dds/DCPS/transport/framework/TheTransportFactory.h>
#include <dds/DCPS/BuiltInTopicUtils.h>
#include "DatabaseNotificationDataReaderListenerImpl.h"


using namespace System;
using namespace System::Threading;

// incomplete types generate LNK4248 warnings when compiled for .NET
SUPPRESS_LNK4248_CORBA
SUPPRESS_LNK4248_TAO


DataServerConnectorState::DataServerConnectorState(CORBA::ORB_ptr orb, Database_ptr database) {
	orb_ = CORBA::ORB::_duplicate(orb);
	database_ = Database::_duplicate(database);
}


DataServerConnector::DataServerConnector() : state_(NULL), thread_(nullptr), startupEvent_(false) {}

DataServerConnector::~DataServerConnector() {
	delete state_;
}

void DataServerConnector::ThreadStart(Object^ param) { 
	((DataServerConnector^)param)->Run();
}

void DataServerConnector::Start() {
	ParameterizedThreadStart^ pts = gcnew ParameterizedThreadStart(&DataServerConnector::ThreadStart);
	thread_ = gcnew Thread(pts);
	thread_->Start(this);
	startupEvent_.WaitOne();
}

void DataServerConnector::Shutdown() {
	if (state_)
		state_->OrbPtr()->shutdown();
	//thread_->Join();
}


DDS::DomainId_t DATABASENOTIFICATION_DOMAIN_ID = 1066;
const char* DATABASENOTIFICATION_TYPE = "DatabaseNotification Type";
const char* DATABASENOTIFICATION_TOPIC = "DatabaseNotification Topic";

void DataServerConnector::Run() {
	DDS::DomainParticipantFactory_var dpf = DDS::DomainParticipantFactory::_nil();
	DDS::DomainParticipant_var participant = DDS::DomainParticipant::_nil();

	int argc = 0, argc2 = 0;
	wchar_t **argv = NULL, **argv2 = NULL;

	try {
		// convert .NET arguments to standard argc/argv
		// duplicate so each ORB has its own set
		array<String^>^ arguments = Environment::GetCommandLineArgs();
		argc = arguments->Length;
		argc2 = argc;
		argv = new wchar_t *[argc];
		argv2 = new wchar_t *[argc2];
		for (int i=0; i<argc; i++) {
			pin_ptr<const wchar_t> arg = PtrToStringChars(arguments[i]);
			argv[i] = _wcsdup(arg);
			argv2[i] = _wcsdup(arg);
		}

		// *** FOR OpenDDS *** 

		// create the participant factory
		dpf = TheParticipantFactoryWithArgs(argc, argv);

		// create the participant
		participant = dpf->create_participant(
			DATABASENOTIFICATION_DOMAIN_ID,
			PARTICIPANT_QOS_DEFAULT,
			DDS::DomainParticipantListener::_nil());
		if (CORBA::is_nil(participant.in())) 
			throw std::exception("create_participant failed");

		// initialize the transport - the transport ID matches the config file
		// must be done after create_participant
		const OpenDDS::DCPS::TransportIdType TRANSPORT_IMPL_ID = 1;
		OpenDDS::DCPS::TransportImpl_rch trans_impl =
			TheTransportFactory->create_transport_impl(TRANSPORT_IMPL_ID, 
			OpenDDS::DCPS::AUTO_CONFIG);

		// create the subscriber
		DDS::Subscriber_var sub =
			participant->create_subscriber(SUBSCRIBER_QOS_DEFAULT,
			DDS::SubscriberListener::_nil());
		if (CORBA::is_nil(sub.in())) 
			throw std::exception("create_subscriber failed");

		// attach the subscriber to the transport
		OpenDDS::DCPS::SubscriberImpl* sub_impl =
			dynamic_cast<OpenDDS::DCPS::SubscriberImpl*>(sub.in());
		if (0 == sub_impl) 
			throw std::exception("cannot obtain the subscriber servant");
		OpenDDS::DCPS::AttachStatus status = sub_impl->attach_transport(trans_impl.in());
		if (status != OpenDDS::DCPS::ATTACH_OK) {
			std::string msg("Cannot attach to the transport: ");
			switch (status) {
				case OpenDDS::DCPS::ATTACH_BAD_TRANSPORT:
					throw std::exception((msg+"ATTACH_BAD_TRANSPORT").c_str());
				case OpenDDS::DCPS::ATTACH_ERROR:
					throw std::exception((msg+"ATTACH_ERROR").c_str());
				case OpenDDS::DCPS::ATTACH_INCOMPATIBLE_QOS:
					throw std::exception((msg+"ATTACH_INCOMPATIBLE_QOS").c_str());
				default:
					throw std::exception((msg+"unknown status").c_str());
			}
		}

		// register the type
		DatabaseNotificationTypeSupport_var databaseNotification_servant 
			= new DatabaseNotificationTypeSupportImpl();
		if (DDS::RETCODE_OK != databaseNotification_servant->register_type(participant.in(),
			DATABASENOTIFICATION_TYPE)) 
			throw std::exception("register_type failed");

		// get the default topic QOS
		DDS::TopicQos default_topic_qos;
		participant->get_default_topic_qos(default_topic_qos);

		// create the topic
		DDS::Topic_var databaseNotification_topic =
			participant->create_topic(DATABASENOTIFICATION_TOPIC, DATABASENOTIFICATION_TYPE,
			default_topic_qos, DDS::TopicListener::_nil());
		if (CORBA::is_nil(databaseNotification_topic.in())) 
			throw std::exception("create_topic failed");

		// create the listener
		DDS::DataReaderListener_var databaseNotification_listener(new DatabaseNotificationDataReaderListenerImpl(this));
		if (CORBA::is_nil(databaseNotification_listener.in())) 
			throw std::exception("cannot create the listener");

		// get the default data reader QOS
		DDS::DataReaderQos dr_default_qos;
		sub->get_default_datareader_qos(dr_default_qos);

		// create the data reader
		DDS::DataReader_var databaseNotification_dr = 
			sub->create_datareader(databaseNotification_topic.in (),
			dr_default_qos,
			databaseNotification_listener.in());
		if (CORBA::is_nil(databaseNotification_dr.in()))
			throw std::exception("create_datareader failed");


		// *** FOR TAO ***

		// initialize the ORB
		CORBA::ORB_var orb = CORBA::ORB_init(argc2, argv2);

		// USE ORBINITREF - start server with -ORBListenEndpoints iiop://machine:port
		// and client with -ORBInitRef DataServer=corbaloc:iiop:machine:port/DataServer
		CORBA::Object_var database_obj = orb->resolve_initial_references("DataServer");

		if (CORBA::is_nil(database_obj.in())) 
			throw std::exception("Could not get Database IOR");

		// narrow the IOR to a DataServer object reference.
		Database_var database = Database::_narrow(database_obj.in());
		if (CORBA::is_nil(database.in())) {
			std::cerr << "IOR was not a Database object reference." << std::endl;
			return;
		}

		// save the references via a pointer to an unmanaged class
		state_ = new DataServerConnectorState(orb, database);

		// good to go - tell the outside world
		startupEvent_.Set();

		// run the ORB - need to wait anyway for DDS, so may as well process AMI calls, etc.
		orb->run();
		orb->destroy();
	}
	catch (CORBA::Exception& ex) {
		std::stringstream ss;
		ss << "Exception: " << ex;
		throw gcnew DataConnectorException(gcnew String(ss.str().c_str()));
	}
	catch (std::exception& ex) {
		std::stringstream ss;
		ss << "Exception: " << ex.what();
		throw gcnew DataConnectorException(gcnew String(ss.str().c_str()));
	}

	// clean up DDS
	try {
		if (!CORBA::is_nil(participant.in()))
			participant->delete_contained_entities();
		if (!CORBA::is_nil(dpf.in()))
			dpf->delete_participant(participant.in ());
	} catch (CORBA::Exception& e) {
		std::stringstream ss;
		ss << "Exception during cleanup: " << e;
		throw gcnew DataConnectorException(gcnew String(ss.str().c_str()));
	}
	TheTransportFactory->release();
	TheServiceParticipant->shutdown();
}


// *** .NET wrappers for CORBA methods ***
// DataConnector must have a pure .NET interface, so arguments must be CLR types

bool DataServerConnector::CreateItem(String ^description, Int64 %id) {
	try {
		pin_ptr<const wchar_t> cppDescription = PtrToStringChars(description);
		CORBA::WString_var desc = CORBA::wstring_dup(cppDescription);
		CORBA::LongLong cid;
		CORBA::Boolean result = state_->DatabasePtr()->CreateItem(desc, cid);
		id = cid;
		return result;
	} catch (CORBA::Exception &ex) {
		std::stringstream ss;
		ss << "Exception: " << ex;
		throw gcnew DataConnectorException(gcnew String(ss.str().c_str()));
	}
}

bool DataServerConnector::ReadItem(Int64 id, String^% description) {
	try {
		CORBA::WString_var desc;
		CORBA::Boolean result = state_->DatabasePtr()->ReadItem(id, desc);
		if (result)
			description = gcnew String(desc.in());
		return result;
	} catch (CORBA::Exception &ex) {
		std::stringstream ss;
		ss << "Exception: " << ex;
		throw gcnew DataConnectorException(gcnew String(ss.str().c_str()));
	}
}

bool DataServerConnector::UpdateItem(Int64 id, String^ description) {
	try {
		pin_ptr<const wchar_t> cppDescription = PtrToStringChars(description);
		CORBA::WString_var desc = CORBA::wstring_dup(cppDescription);
		return state_->DatabasePtr()->UpdateItem(id, desc);
	} catch (CORBA::Exception &ex) {
		std::stringstream ss;
		ss << "Exception: " << ex;
		throw gcnew DataConnectorException(gcnew String(ss.str().c_str()));
	}
}

bool DataServerConnector::DeleteItem(Int64 id) {
	try {
		return state_->DatabasePtr()->DeleteItem(id);
	} catch (CORBA::Exception &ex) {
		std::stringstream ss;
		ss << "Exception: " << ex;
		throw gcnew DataConnectorException(gcnew String(ss.str().c_str()));
	}
}

