#include <iostream>
#include <fstream>
#include <string>
#include <tchar.h>
#include <tao/IORTable/IORTable.h>
#include "Database_i.h"
#include "Shared/LNK4248.h"

// for DDS
#include "DatabaseNotification/DatabaseNotificationTypeSupportImpl.h"
#include <dds/DCPS/Service_Participant.h>
#include <dds/DCPS/Marked_Default_Qos.h>
#include <dds/DCPS/PublisherImpl.h>
#include <dds/DCPS/transport/framework/TheTransportFactory.h>

using namespace System;

// incomplete types generate LNK4248 warnings when compiled for .NET
SUPPRESS_LNK4248_CORBA

// to allow the server to be shut down gracefully with ctrl-c
CORBA::ORB_ptr gOrb_ = NULL;
static BOOL WINAPI ControlCHandler(DWORD /*type*/) {
	std::cout << "Quitting because of ctrl-c" << std::endl;
	if (gOrb_)
		gOrb_->shutdown();
	return TRUE;
}


DDS::DomainId_t DATABASENOTIFICATION_DOMAIN_ID = 1066;
const char* DATABASENOTIFICATION_TYPE = "DatabaseNotification Type";
const char* DATABASENOTIFICATION_TOPIC = "DatabaseNotification Topic";


int ACE_TMAIN(int argc, ACE_TCHAR *argv[]) {
	DDS::DomainParticipantFactory_var dpf = DDS::DomainParticipantFactory::_nil();
	DDS::DomainParticipant_var participant = DDS::DomainParticipant::_nil();

	try {
		// to allow the server to be shut down gracefully with ctrl-c
		SetConsoleCtrlHandler(ControlCHandler, TRUE);

		// must duplicate argc/argv as each ORB absorbs parameters
		int argcDDS = 0;
		ACE_TCHAR **argvDDS=new ACE_TCHAR *[argc];
		for (int i=0; i<argc; i++)
			if (!ACE_OS::strcmp(ACE_TEXT("-ORBListenEndpoints"), argv[i]))
				i++;  // don't pass -ORBListenEndpoints and its argument to DDS
			else
				argvDDS[argcDDS++] = ACE_OS::strdup(argv[i]);

		// *** FOR OpenDDS *** 

		// create the participant factory
		dpf = TheParticipantFactoryWithArgs(argcDDS, argvDDS);

		// create the participant
		participant = dpf->create_participant(
			DATABASENOTIFICATION_DOMAIN_ID,
			PARTICIPANT_QOS_DEFAULT,
			DDS::DomainParticipantListener::_nil());
		if (CORBA::is_nil(participant.in())) 
			throw std::exception("create_participant failed");

		// initialize the transport - the transport ID matches the config file
		// must be done after create_participant
		const OpenDDS::DCPS::TransportIdType TRANSPORT_IMPL_ID = 1;
		OpenDDS::DCPS::TransportImpl_rch trans_impl =
			TheTransportFactory->create_transport_impl(TRANSPORT_IMPL_ID, 
			OpenDDS::DCPS::AUTO_CONFIG);

		// create the publisher
		DDS::Publisher_var pub = participant->create_publisher(PUBLISHER_QOS_DEFAULT,
			DDS::PublisherListener::_nil());
		if (CORBA::is_nil(pub.in())) 
			throw std::exception("create_publisher failed");

		// attach the publisher to the transport
		OpenDDS::DCPS::PublisherImpl* pub_impl =
			dynamic_cast< OpenDDS::DCPS::PublisherImpl* >(pub.in());
		if (0 == pub_impl)
			throw std::exception("cannot obtain the publisher servant");
		OpenDDS::DCPS::AttachStatus status = pub_impl->attach_transport(trans_impl.in());
		if (status != OpenDDS::DCPS::ATTACH_OK) {
			std::string msg("Cannot attach to the transport: ");
			switch (status) {
				case OpenDDS::DCPS::ATTACH_BAD_TRANSPORT:
					throw std::exception((msg+"ATTACH_BAD_TRANSPORT").c_str());
				case OpenDDS::DCPS::ATTACH_ERROR:
					throw std::exception((msg+"ATTACH_ERROR").c_str());
				case OpenDDS::DCPS::ATTACH_INCOMPATIBLE_QOS:
					throw std::exception((msg+"ATTACH_INCOMPATIBLE_QOS").c_str());
				default:
					throw std::exception((msg+"unknown status").c_str());
			}
		}

		// register the type
		DatabaseNotificationTypeSupport_var databaseNotification_servant 
			= new DatabaseNotificationTypeSupportImpl();
		if (DDS::RETCODE_OK != databaseNotification_servant->register_type(participant.in(), 
			DATABASENOTIFICATION_TYPE))
			throw std::exception("register_type failed");

		// get the default topic QOS
		DDS::TopicQos default_topic_qos;
		participant->get_default_topic_qos(default_topic_qos);

		// create the topic
		DDS::Topic_var databaseNotification_topic =
		participant->create_topic(DATABASENOTIFICATION_TOPIC, DATABASENOTIFICATION_TYPE,
			default_topic_qos, DDS::TopicListener::_nil());
		if (CORBA::is_nil(databaseNotification_topic.in())) 
			throw std::exception("create_topic failed");

		// get the default data writer QOS
		DDS::DataWriterQos dw_default_qos;
		pub->get_default_datawriter_qos(dw_default_qos);

		// create the data writer
		DDS::DataWriter_var databaseNotification_base_dw =
			pub->create_datawriter(databaseNotification_topic.in(),
			dw_default_qos,
			DDS::DataWriterListener::_nil());
		if (CORBA::is_nil(databaseNotification_base_dw.in()))
			throw std::exception("create_datawriter failed");

		DatabaseNotificationDataWriter_var databaseNotification_dw
			= DatabaseNotificationDataWriter::_narrow(databaseNotification_base_dw.in());
		if (CORBA::is_nil(databaseNotification_dw.in())) 
			throw std::exception("DatabaseNotificationDataWriter could not be narrowed");


		// *** FOR TAO ***

		// initialize the ORB
		CORBA::ORB_var orb = CORBA::ORB_init(argc, argv);
		gOrb_ = orb.in();

		// get a reference to the RootPOA
		CORBA::Object_var obj = orb->resolve_initial_references("RootPOA");
		PortableServer::POA_var poa = PortableServer::POA::_narrow(obj.in());

		// activate the POAManager
		PortableServer::POAManager_var mgr = poa->the_POAManager();
		mgr->activate();

		// open the database - DataServer (for now) maintains one open database connection for all clients
		DataLib::Database database;
		if (!database.Open())
			throw std::exception("cannot open the database");

		// create the database servant
		Database_i servant(%database, databaseNotification_dw.in());
		PortableServer::ObjectId_var oid = poa->activate_object(&servant);
		CORBA::Object_var database_obj = poa->id_to_reference(oid.in());

		// for corbaloc - start server as -ORBListenEndpoints iiop://machine:port
		CORBA::String_var ior_str = orb->object_to_string(database_obj.in());
		CORBA::Object_var tobj = orb->resolve_initial_references("IORTable");
		IORTable::Table_var table = IORTable::Table::_narrow(tobj.in());
		table->bind("DataServer", ior_str.in());
		std::cout << "DataServer bound to IORTable" << std::endl;

		// accept requests from clients
		orb->run();
		orb->destroy();
	}
	catch (CORBA::Exception& ex) {
		std::cerr << "CORBA exception: " << ex << std::endl;
	}
	catch (std::exception& ex) {
		std::cerr << "Exception: " << ex.what() << std::endl;
	}

	// clean up DDS
	try {
		if (!CORBA::is_nil(participant.in()))
			participant->delete_contained_entities();
		if (!CORBA::is_nil(dpf.in()))
			dpf->delete_participant(participant.in());
	} catch (CORBA::Exception& e) {
		std::cerr << "Exception during cleanup: " << e << std::endl;
	}
	TheTransportFactory->release();
	TheServiceParticipant->shutdown();

	return 0;
}