package com.ociweb.tnb;
import android.content.Context;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Matrix;
import android.graphics.Paint;
import android.util.AttributeSet;
import android.util.Log;
import android.view.*;
import android.widget.ImageView;

public class VectorView extends ImageView
{
    private static final String LOG_TAG = "Sensors:Vector View";

    private float mX = 0.0f;
    private float mY = 0.0f;
    private float mZ = 0.0f;

    private static final double EPSILON = 0.001;
    private float mScaleX = 0.1f;
    private float mScaleY = 0.1f;
    private double mScaleZ = 0.1f;
    private float mNormX;
    private float mNormY;
    private double mNormZ;

    public VectorView(Context context) {
        super(context);
    }

    public VectorView(Context context, AttributeSet attributeSet) {
        super(context, attributeSet);
/* To add properties in the layout file, add a 'styleable' resource, and this code:
        TypedArray attributes = context.obtainStyledAttributes(
                attributeSet,
                R.styleable.VectorView);
        mSomeProperty = attributes.getInteger(R.styleable.VectorView_some_property, -1);
        attributes.recycle();
*/
    }
    public void setVector(float x, float y, float z)
    {
        mX = x;
        if(Math.abs(mX) > mScaleX)
        {
            mScaleX = Math.abs(mX);
        }
        mY = y;
        if(Math.abs(mY) > mScaleY)
        {
            mScaleY = Math.abs(mY);
        }
        mZ = z;
        if(Math.abs(mZ) > mScaleZ)
        {
            mScaleZ = Math.abs(mZ);
        }
        invalidate();
    }
    public void setScale(float x, float y, float z)
    {
        mScaleX = x;
        mScaleY = y;
        mScaleZ = z;
    }

    private void normalizePoints()
    {
        mNormX = mX / mScaleX;
        mNormY = mY / mScaleY;
        mNormZ = mZ / mScaleZ;
    }

    private void orientPoints() {
        // Rotate coordinates around the Z axis to cope with device orientation
        // screen coordinates change with device orientation, but sensor coordinates do not.
        // Note that the device 'y' coordinate has positive numbers up, but the 2D graphics
        // package has positive numbers down, so the y axis is negated in the native orientation
        Display display = ((WindowManager) getContext().getSystemService(Context.WINDOW_SERVICE)).getDefaultDisplay();
        int orientation = display.getOrientation();
        float tempX = mNormX;
        switch(orientation)
        {
            default:
            case Surface.ROTATION_0:
//                mNormX = mNormX;
                mNormY = -mNormY;
                break;
            case Surface.ROTATION_90:
                mNormX = -mNormY;
                mNormY = -tempX;
                break;
            case Surface.ROTATION_180:
                mNormX = -mNormX;
//                mNormY = mNormY;
                break;
            case Surface.ROTATION_270:
                mNormX = mNormY;
                mNormY = tempX;
                break;
        }
    }

    @Override
    protected void onDraw(Canvas canvas) {
        super.onDraw(canvas);

        Log.i(LOG_TAG, "onDraw: (" + mX + "," + mY + ',' + mZ + "): [" + mScaleX + ',' + mScaleY + ',' + mScaleZ + ']');
        // scale the points to the range (-1, +1)
        normalizePoints();
        // then swap axes as necessary to match the orientation of the device
        orientPoints();
        Log.i(LOG_TAG, "Normalized: " + mNormX + "," + mNormY + ',' + mNormZ + ")");
        // Note that the canvas comes preconfigured with a matrix
        // that maps coordinates in the rectangle (0,0,width, height) into actual screen coordinates
        // Messing with this matrix is hairy, so I scale the points
        // I want to plot into these coordinates.
        int width = this.getWidth();
        int height = this.getHeight();
        int minDim = Math.min(width, height);
        int originX = width/2;
        int originY = height/2;
        int scaledX = originX + (int)(mNormX * (float)minDim * 0.5);
        int scaledY = originY + (int)(mNormY * (float)minDim * 0.5);
        int radius = (int)((mNormZ + 1.2) * minDim * 0.02);
        Log.i(LOG_TAG, "Scaled to ("
                + width + ',' + height + "::"
                + mScaleX + ',' + mScaleY + ")->"
                + scaledX + ',' + scaledY + ')'
                + minDim
            );
        Log.i(LOG_TAG, "Matrix: " + canvas.getMatrix().toShortString());

        Paint paint = new Paint();
        int zFactor = (int)(mNormZ * 128.0); // range: (-128, +128)
        // Draw a box on the screen to indicate the expected limits of the values
        paint.setColor(Color.DKGRAY);
        canvas.drawRect(originX - minDim/2, originY - minDim/2, originX + minDim/2, originY + minDim/2, paint);

        // Now draw a line from the origin to the X,Y point
        // The Z value determines the color of that line
        paint.setColor(Color.rgb(128 + zFactor, 128, 128 - zFactor));
        canvas.drawLine(originX, originY, scaledX, scaledY, paint);
        // and put a filled circle at the end of that line.
        canvas.drawCircle(scaledX, scaledY, radius, paint);
    }
}
