/*
 * Copyright (c) 2011, Object Computing, Inc.
 * All Rights reserved
 * See the file license.txt for licensing information.
 */
package com.ociweb.tnb;

import android.app.Activity;
import android.content.Intent;
import android.hardware.Sensor;
import android.hardware.SensorEvent;
import android.hardware.SensorEventListener;
import android.hardware.SensorManager;
import android.os.Bundle;
import android.os.Handler;
import android.view.Menu;
import android.view.MenuItem;
import android.widget.TextView;

import java.util.List;

/**
 * SensorDisplay is an Android Activity that displays information about a particular sensor.
 * The sensor is chosen via an Extra value in the Intent used to start this activity
 *   SENSOR_NAME= {the name of the sensor as returned by the SensorManager}
 */
public class SensorDisplay extends Activity {
    public static final String SENSOR_NAME = "SENSOR_NAME";
    public static final int MENU_DISPLAY_VECTOR = 1;

    private Sensor mSensor;
    private SensorManager mSensorManager;

    private TextView mSensorAccuracyCtrl;
    private TextView mSensorValuesCtrl;
    private TextView mSensorTimestampCtrl;

    private long mTimestamp = 0;
    private int mAccuracyChanged = -1;
    private int mAccuracy = -1;
    private int mValueCount;
    private float[] mValues;
    private String[] mValueNames;
    private String mUnit;

    private SensorEventListener mEventListener;

    private Handler mHandler = new Handler();

    /**
     * The onSensorChanged() method may be called in a different thread from the one that manages
     * the UI.  Repainting the screen can be deferred to the UI thread using this Runnable.
     */
    private Runnable mUpdateUI = new Runnable(){
        public void run() {
            repaint();
        }
    };
    private String mSensorName;

    @Override
    public void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.display_sensor);
        // find the controls that will be updated dynamically

        mSensorAccuracyCtrl = (TextView)findViewById(R.id.sensor_accuracy);
        mSensorTimestampCtrl = (TextView)findViewById(R.id.sensor_timestamp);
        mSensorValuesCtrl = (TextView)findViewById(R.id.sensor_values);

        mSensorName = getIntent().getExtras().getString(SENSOR_NAME);
        mSensorManager = (SensorManager) getSystemService(SENSOR_SERVICE);
        List<Sensor> sensorList = mSensorManager.getSensorList(Sensor.TYPE_ALL);
        for (Sensor sensor : sensorList) {
            if (sensor.getName().equals(mSensorName)) {
                mSensor = sensor;
                break;
            }
        }
        if(mSensor == null)
        {
            TextView sensorNameCtrl = (TextView) findViewById(R.id.sensor_name);
            sensorNameCtrl.setText(getResources().getString(R.string.unknown_sensor));
        }
        else
        {
            TextView sensorNameCtrl = (TextView) findViewById(R.id.sensor_name);
            sensorNameCtrl.setText(getResources().getString(R.string.sensor_name) + mSensor.getName());

            TextView sensorTypeCtrl = (TextView) findViewById(R.id.sensor_type);
            String sensorType = getResources().getString(R.string.unknown_sensor_type);
            switch(mSensor.getType())
            {
                case Sensor.TYPE_ACCELEROMETER:
                    sensorType = getResources().getString(R.string.sensor_type_accelerometer);
                    mValueCount = 3;
                    mValues = new float[mValueCount];
                    mValueNames = new String[mValueCount];
                    mValueNames[0] = getResources().getString(R.string.x_axis);
                    mValueNames[1] = getResources().getString(R.string.y_axis);
                    mValueNames[2] = getResources().getString(R.string.z_axis);
                    mUnit = getResources().getString(R.string.accelerator_units);
                    break;
                case Sensor.TYPE_LIGHT:
                    sensorType = getResources().getString(R.string.sensor_type_light);
                    mValueCount = 1;
                    mValues = new float[mValueCount];
                    mValueNames = new String[mValueCount];
                    mValueNames[0] = getResources().getString(R.string.ambient);
                    mUnit = getResources().getString(R.string.light_units);
                    break;
                case Sensor.TYPE_PRESSURE:
                    sensorType = getResources().getString(R.string.sensor_type_pressure);
                    mValueCount = 1;
                    mValues = new float[mValueCount];
                    mValueNames = new String[mValueCount];
                    mValueNames[0] = getResources().getString(R.string.ambient);
                    break;
                case Sensor.TYPE_GYROSCOPE:
                    sensorType = getResources().getString(R.string.sensor_type_gyroscope);
                    mValueCount = 3;
                    mValues = new float[mValueCount];
                    mValueNames = new String[mValueCount];
                    mValueNames[0] = getResources().getString(R.string.x_axis);
                    mValueNames[1] = getResources().getString(R.string.y_axis);
                    mValueNames[2] = getResources().getString(R.string.z_axis);
                    mUnit = getResources().getString(R.string.gyroscope_units);
                    break;
                case Sensor.TYPE_PROXIMITY:
                    sensorType = getResources().getString(R.string.sensor_type_proximity);
                    mValueCount = 1;
                    mValues = new float[mValueCount];
                    mValueNames = new String[mValueCount];
                    mValueNames[0] = getResources().getString(R.string.distance);
                    mUnit = getResources().getString(R.string.proximity_units);
                    break;
                case Sensor.TYPE_MAGNETIC_FIELD:
                    sensorType = getResources().getString(R.string.sensor_type_magnetic);
                    mValueCount = 3;
                    mValues = new float[mValueCount];
                    mValueNames = new String[mValueCount];
                    mValueNames[0] = getResources().getString(R.string.x_axis);
                    mValueNames[1] = getResources().getString(R.string.y_axis);
                    mValueNames[2] = getResources().getString(R.string.z_axis);
                    mUnit = getResources().getString(R.string.magnetic_units);
                    break;
                case Sensor.TYPE_ORIENTATION:
                    sensorType = getResources().getString(R.string.sensor_type_orientation);
                    mValueCount = 3;
                    mValues = new float[mValueCount];
                    mValueNames = new String[mValueCount];
                    mValueNames[0] = getResources().getString(R.string.x_axis);
                    mValueNames[1] = getResources().getString(R.string.y_axis);
                    mValueNames[2] = getResources().getString(R.string.z_axis);
                    mUnit = getResources().getString(R.string.orientation_unit);
                    break;
                case Sensor.TYPE_TEMPERATURE:
                    sensorType = getResources().getString(R.string.sensor_type_temperature);
                    mValueCount = 1;
                    mValues = new float[mValueCount];
                    mValueNames = new String[mValueCount];
                    mValueNames[0] = getResources().getString(R.string.ambient);
                    mUnit = getResources().getString(R.string.temperature_units);
                    break;
            }
            sensorTypeCtrl.setText(getResources().getString(R.string.type) + sensorType);

            TextView sensorVendorCtrl = (TextView) findViewById(R.id.sensor_vendor);
            sensorVendorCtrl.setText(getResources().getString(R.string.vendor) + mSensor.getVendor());

            TextView sensorVersionCtrl = (TextView) findViewById(R.id.sensor_version);
            sensorVersionCtrl.setText(getResources().getString(R.string.version) + mSensor.getVersion());

            TextView sensorPowerCtrl = (TextView) findViewById(R.id.sensor_power);
            sensorPowerCtrl.setText(getResources().getString(
                    R.string.uses) + mSensor.getPower() + getResources().getString(R.string.current_units));

            TextView sensorResolutionCtrl = (TextView) findViewById(R.id.sensor_resolution);
            sensorResolutionCtrl.setText(getResources().getString(R.string.resolution) + mSensor.getResolution());

            TextView sensorMaxRangeCtrl = (TextView) findViewById(R.id.sensor_max_range);
            sensorMaxRangeCtrl.setText(getResources().getString(R.string.max_range) + mSensor.getMaximumRange());

            TextView mSensorUnitsCtrl = (TextView) findViewById(R.id.sensor_units);
            mSensorUnitsCtrl.setText(getResources().getString(R.string.units) + mUnit);

            TextView mSensorValueCountCtrl = (TextView) findViewById(R.id.sensor_value_count);
            mSensorValueCountCtrl.setText(getResources().getString(R.string.number_of_values) + mValueCount);
        }
    }

    private void repaint() {
        mSensorAccuracyCtrl.setText(getResources().getString(R.string.accuracy) + mAccuracy + '(' + mAccuracyChanged + ')');
        mSensorTimestampCtrl.setText(getResources().getString(R.string.timestamp) + mTimestamp);
        StringBuffer valueText = new StringBuffer();
        for (int nValue = 0; nValue < mValueCount; nValue++) {
            valueText.append(mValueNames[nValue])
                    .append(": ")
                    .append(mValues[nValue])
                    .append("\n");
        }
        mSensorValuesCtrl.setText(valueText.toString());
    }

    @Override
    public boolean onCreateOptionsMenu(Menu menu)
    {
        menu.add(0, MENU_DISPLAY_VECTOR , 0, R.string.menu_display_vector);
        return super.onCreateOptionsMenu(menu);
    }

    @Override
    public boolean onMenuItemSelected(int featureId, MenuItem item)
    {
        switch (item.getItemId())
        {
        case MENU_DISPLAY_VECTOR:
            Intent i = new Intent(this, VectorDisplay.class);
            i.putExtra(VectorDisplay.SENSOR_NAME, mSensorName);
            startActivity(i);
            return true;
        }

        return super.onMenuItemSelected(featureId, item);
    }


    @Override
    protected void onResume() {
        super.onResume();
        if(mSensor != null)
        {
            mEventListener = new SensorEventListener(){
                        public void onSensorChanged(SensorEvent sensorEvent) {
                            mAccuracy = sensorEvent.accuracy;
                            mTimestamp = sensorEvent.timestamp;
                            System.arraycopy(sensorEvent.values, 0, mValues, 0, mValueCount);
                            mHandler.post(mUpdateUI);
                        }
                        public void onAccuracyChanged(Sensor sensor, int i) {
                            mAccuracyChanged = i;
                            mHandler.post(mUpdateUI);
                        }
                    };

            mSensorManager.registerListener(
                    mEventListener,
                    mSensor,
                    SensorManager.SENSOR_DELAY_UI);
        }
    }

    @Override
    protected void onPause() {
        super.onPause();
        if(mEventListener != null)
        {
            mSensorManager.unregisterListener(mEventListener, mSensor);
            mEventListener = null;
        }
    }
}