class DSL

  # Adds methods to the current class that return their arguments
  # on to their callers. Useful for adding non-processing methods
  # to the DSL that improve the readability of the syntax
  # This idea is described in Jay Fields blog (http://jayfields.blogspot.com)
  def self.bubble(*methods)
    methods.each do |method|
      define_method(method) { |args| args }
    end
  end
end

class Text < DSL
  # Adding these methods allows the DSL to be more English-like.
  bubble :each, :every, :of, :the
  
  def initialize(text)
    @text = text
  end

  # This returns an array containing an element for each first item
  # (word or sentence) that is found.
  # Each element is an array containing two integers.
  # The first is the index where the item begins
  # and the second is the length of the item.
  def first(sentences=nil)
    end_point :first, sentences
  end
  
  # This returns an array containing an element for each last item
  # (word or sentence) that is found.
  # Each element is an array containing two integers.
  # The first is the index where the item begins
  # and the second is the length of the item.
  def last(sentences=nil)
    end_point :last, sentences
  end

  # called in response to a keyword not defined
  # as a method for this class
  def method_missing(method_id, *args)
    raise "don't understand #{method_id}"
  end
  
  # This returns an array containing an element
  # for each sentence that is found.
  # Each element is an array containing two integers.
  # The first is the index where the item begins
  # and the second is the length of the item.
  # Returns an array of sentence strings.
  def sentence
    raise "invalid use of sentence" unless @last_token.nil?

    @last_token = @last_unit = :sentence

    arr = []
    cursor = 0

    @text.split(/\.|\?|!/).each do |sentence|
      sentence.lstrip! # removes leading whitespace
      len = sentence.length + 1 # +1 for punctuation
      arr << [cursor, len]
      cursor += len
      cursor += 1 # for space between sentences
    end

    arr
  end

  # The parameter is an array of arrays.
  # The inner elements can be integer pairs (index and length)
  # or words in a sentence.
  # Returns an array with an element for each sentence in the text.
  # Each element is an array of the words in the corresponding sentence.
  def word(arg=nil)
    raise "invalid use of word" unless arg
    unless [nil, :sentence].include?(@last_unit)
      raise "invalid use of word"
    end

    @last_token = @last_unit = :word

    all_words = []
    
    # Start cursor at index of first word in first sentence passed in.
    cursor = arg[0][0]

    arg.each do |sentence|
      if sentence.instance_of? Array then
        index, length = *sentence
        sentence = @text[index, length]
      end

      all_words << split_by_words(cursor, sentence)
      cursor += sentence.length
      cursor += 1 # for space between sentences
    end 

    all_words
  end

  private
  
  def split_by_words(cursor, sentence)
    words = sentence.split(' ')

    # Remove punctuation from last word in sentence.
    words[-1] = words[-1][0..-2]

    arr = []

    words.each do |word|
      arr << [cursor, word.length]
      cursor += word.length
      cursor += 1 # for space between words
    end

    arr
  end
  
  def end_point(position, sentences=nil)
    raise "invalid use of #{position}" unless sentences
    unless [:sentence, :word].include?(@last_token)
      raise "invalid use of #{position}"
    end

    @last_token = position

    case @last_unit
    when :sentence
      [sentences.send(position)]
    when :word
      words = []
      sentences.each { |sentence| words << sentence.send(position) }
      words
    else
      nil # should never happen
    end
  end
end
