
package com.ociweb.demo;

import javax.jws.Oneway;
import javax.jws.soap.SOAPBinding;
import javax.jws.WebMethod;
import javax.jws.WebParam;
import javax.jws.WebResult;
import javax.jws.WebService;
import javax.xml.ws.Endpoint;
import javax.xml.ws.Holder;

import java.util.Date;
import java.util.Random;

// use-case
// -- client requests estimate with getEstimate() and receives partially-filled 
//    OrderRecord, called "Estimate"
// -- client places order by passing the "Estimate" back as an "Order", which
//    is filled in with order information
// -- client later confirms receipt of shipment via confirmReceipt()

class OrderRecord {
    // very poor design! assume "legacy" record 
    // estimate fields
    public int numUnits;
    public String customerId;
    public float cost;
    // order fields
    public String orderCode;
    public Date expectedArrival;
}

@WebService (
name="AcmeWebOrderInterface"
)
@SOAPBinding(style=SOAPBinding.Style.RPC)
interface LegacyInterface {
    
    @WebMethod
    @Oneway
    void confirmReceipt(String orderCode);
    
    @WebMethod
    @WebResult(name="Estimate")
    OrderRecord getEstimate( String customerId, int numUnits );
    
    @WebMethod
    boolean placeOrder( @WebParam(mode=WebParam.Mode.INOUT) 
                            Holder<OrderRecord> orderHolder );    
}

// Note that this SIB defers meta-data information to the SEI
@WebService(
    endpointInterface="com.ociweb.demo.LegacyInterface",
    serviceName="AcmeWebService",
    portName="AcmeWebOrderPort"
)
public class Acme implements LegacyInterface {

    public void confirmReceipt(String orderCode) {
        // oneway methods should generally spawn a thread to do 
        // the business logic
        System.out.println("writing confirmation to DB for: " + orderCode);
    }
  
    public OrderRecord getEstimate( String customerId,   
                                            int numUnits ) {
        OrderRecord record = new OrderRecord();
        System.out.println("received estimate request. customerId = " + customerId);
        record.numUnits = numUnits;
        record.customerId = customerId;
        record.cost = numUnits * 100.0f;
        return record;
    }

    public boolean placeOrder( Holder<OrderRecord> orderHolder ) {
        OrderRecord order = orderHolder.value;
        final int arbitraryLimit = 10000;
        int randomInt = (new Random()).nextInt(arbitraryLimit);
        order.orderCode = Integer.toString(randomInt); 
        order.expectedArrival = new Date();
        System.out.println("placing order with orderCode = " + order.orderCode);
        return true;
    }
    
    // arbitrary unit-test method 
    boolean testingMethod() { return true; }
    
    public static void main(String[] args) {
        String publishUrl = System.getProperty("publish.url");
        if( publishUrl != null && publishUrl.length() > 0 ) {
            System.out.println("publishing service at: " + publishUrl);
            LegacyInterface acmeServer = new Acme();
            Endpoint endpoint = Endpoint.publish(publishUrl,acmeServer);
        } else {
            System.err.println("usage: AcmeServer publishUrl");
        }
    }
}
