package com.ociweb.demo;

import android.app.Activity;
import android.os.Bundle;
import android.os.Handler;
import android.view.View;
import android.view.View.OnClickListener;
import android.widget.Button;
import android.widget.TextView;
import android.widget.Toast;

public class HomeActivity extends Activity implements OnClickListener, HomeModelListener {

  private Button mStartButton;
  private TextView mStatusLabel;
  
  private static final String ENABLED_KEY = "com.ociweb.buttonEnabled";
  private static final String HOME_MODEL_KEY = "com.ociweb.homeModel";

  // background threads use this Handler to post messages to
  // the main application thread
  private final Handler mHandler = new Handler();
  
  // this data model knows when a thread is fetching data
  private HomeModel mHomeModel;

  // post this to the Handler when the background thread completes
  private final Runnable mUpdateDisplayRunnable = new Runnable() {
    public void run() {
      updateDisplay();
    }
  };

  @Override
  public void onCreate(Bundle savedInstanceState) {
    super.onCreate(savedInstanceState);
    setContentView(R.layout.home);

    mStartButton = (Button) findViewById(R.id.start_background_thread_btn);
    mStatusLabel = (TextView) findViewById(R.id.thread_status_label);

    mStartButton.setOnClickListener(this);
    
    if (savedInstanceState != null) {
      if (savedInstanceState.containsKey(ENABLED_KEY)) {
        mStartButton.setEnabled(savedInstanceState.getBoolean(ENABLED_KEY));
      }
      if (savedInstanceState.containsKey(HOME_MODEL_KEY)) {
        mHomeModel = (HomeModel) savedInstanceState.getSerializable(HOME_MODEL_KEY);
      }
    }
    if (mHomeModel == null) {
      // the first time in, create a new model
      mHomeModel = new HomeModel();
    }
  }

  @Override
  protected void onPause() {
    super.onPause();
    // detach from the model
    mHomeModel.setHomeModelListener(null);
  }

  @Override
  protected void onResume() {
    super.onResume();
    // attach to the model
    mHomeModel.setHomeModelListener(this);
    
    // synchronize the display, in case the thread completed
    // while this activity was not visible. For example, if
    // a phone call occurred while the thread was running.
    updateDisplay();
  }

  public void homeModelChanged(HomeModel hm) {
    // this may be called from a background thread, so post
    // to the handler
    mHandler.post(mUpdateDisplayRunnable);
  }

  @Override
  protected void onSaveInstanceState(Bundle outState) {
    super.onSaveInstanceState(outState);
    outState.putBoolean(ENABLED_KEY, mStartButton.isEnabled());
    outState.putSerializable(HOME_MODEL_KEY, mHomeModel);
  }

  public void onClick(View v) {
    if (v == mStartButton) {
      new DataFetcherThread(mHomeModel).start();
    }
  }

  private void updateDisplay() {
    mStartButton.setEnabled(mHomeModel.getState() != HomeModel.State.RUNNING);
    
    switch (mHomeModel.getState()) {
    case RUNNING:
      mStatusLabel.setText(R.string.thread_running);
      Toast.makeText(this, R.string.thread_running, Toast.LENGTH_SHORT);
      break;
    case NOT_STARTED:
      mStatusLabel.setText(R.string.thread_not_started);
      break;
    case FINISHED:
      mStatusLabel.setText(R.string.thread_finished);
      Toast.makeText(this, R.string.thread_finished, Toast.LENGTH_SHORT);
      break;
    }
  }
}