package com.ociweb.jnb.feb2009;

import com.google.common.base.Function;
import com.google.common.base.Functions;
import com.google.common.base.Join;
import com.google.common.collect.*;

import static com.ociweb.jnb.feb2009.FindBloodDonors.BloodType.*;

import java.util.*;


public class FindBloodDonors {
    enum BloodType {
        OPOS, APOS, BPOS, ABPOS, ONEG, ANEG, BNEG, ABNEG
    }

    public static void main(String[] args) {

        // donor -> recipient red blood cell compatibility (source = http://en.wikipedia.org/wiki/Blood_type)
        Multimap<BloodType, BloodType> donorToRecipients = new ImmutableMultimap.Builder<BloodType, BloodType>()
            .putAll(ONEG, ONEG, OPOS, ANEG, APOS, BNEG, BPOS, ABNEG, ABPOS)
            .putAll(OPOS, OPOS, APOS, BPOS, ABPOS)
            .putAll(ANEG, ANEG, APOS, ABNEG, ABPOS)
            .putAll(APOS, APOS, ABPOS)
            .putAll(BNEG, BNEG, BPOS, ABNEG, ABPOS)
            .putAll(BPOS, BPOS, ABPOS).putAll(ABNEG, ABNEG, ABPOS)
            .putAll(ABPOS, ABPOS)
            .build();

        // recipient -> donor red blood cell compatibility
        Multimap<BloodType, BloodType> recipientToDonor = Multimaps.inverseHashMultimap(donorToRecipients);

        // cast member -> blood type   N -> 1
        Map<String, BloodType> castBloodTypes = new ImmutableMap.Builder<String, BloodType>()
                .put("Fred", ONEG)
                .put("Wilma", APOS)
                .put("Pebbles", ANEG)
                .put("Barney", ABPOS)
                .put("Betty", OPOS)
                .put("Bamm-Bamm", ANEG)
                .build();

        Multimap<BloodType, String> castBloodTypesInv = Multimaps.inverseHashMultimap(Multimaps.forMap(castBloodTypes));

        // Barney cuts his finger and needs blood.  Who from the cast can donate?
        String injured = "Barney";

        // create an empty set to hold the possible donors for later display
        Set<String> possibleDonors = new HashSet<String>();

        // look up the blood type of the injured person
        BloodType injuredType = castBloodTypes.get(injured);

        // look up the compatible donor types for the injured person's blood type
        Collection<BloodType> compatibleDonorTypes = recipientToDonor.get(injuredType);

        // iterate over each compatible donor bood type
        for (BloodType compatibleDonorType : compatibleDonorTypes) {

            // find cast members with the compatible type
            Collection<String> castMembersWithType = castBloodTypesInv.get(compatibleDonorType);

            // add all cast members with the compatible type to the set of possible donors
            possibleDonors.addAll(castMembersWithType);
        }

        possibleDonors.remove(injured); // the donor shouldn't try to donate to him/herself
        possibleDonors.remove("Pebbles"); // too young to donate
        possibleDonors.remove("Bamm-Bamm"); // too young to donate

        System.out.printf("%s can donate to %s\n", Join.join(", ", possibleDonors), injured);

        // blood type -> percentage in USA population (source = http://en.wikipedia.org/wiki/Blood_type)
        Map<BloodType, Double> bloodTypeDistributionUSA = new ImmutableMap.Builder<BloodType, Double>()
            .put(OPOS, 37.4)
            .put(APOS, 35.7)
            .put(BPOS, 8.5)
            .put(ABPOS, 3.4)
            .put(ONEG, 6.6)
            .put(ANEG, 6.3)
            .put(BNEG, 1.5)
            .put(ABNEG, 0.6)
            .build();

        // create a Function that looks up the percentage in population based on Blood Type
        Function<BloodType, Double> bloodTypePct = Functions.forMap(bloodTypeDistributionUSA);

        // create a Function that looks up the Blood Type for a person
        Function<String, BloodType> personToBloodType = Functions.forMap(castBloodTypes);

        // create a Function that looks up the percentage in population with the same Blood Type as a given person
        Function<String, Double> personToBloodTypePct = Functions.compose(bloodTypePct, personToBloodType);

        // user our composed lookups to create a Comparator that sorts by availability
        Comparator<String> personToBloodTypePctComparator = Comparators.fromFunction(personToBloodTypePct);

        // reverse the comparator to prefer blood types with higher availability (descending)
        Comparator<String> personToBloodTypePctComparatorDescend =
                Collections.reverseOrder(personToBloodTypePctComparator);

        // create a Set of donors that is sorted according to blood type availability, favoring higher availability
        Set<String> possibleDonorsOptimized = Sets.newTreeSet(personToBloodTypePctComparatorDescend, possibleDonors);

        System.out.printf("%s (sorted in order of preference) can donate to %s\n",
                Join.join(", ", possibleDonorsOptimized), injured);
    }
}
