package com.ociweb.jnb.jun2010.jbtrek;

import org.fest.swing.core.BasicRobot;
import org.fest.swing.core.GenericTypeMatcher;
import org.fest.swing.core.Robot;
import org.fest.swing.finder.JOptionPaneFinder;
import org.fest.swing.finder.WindowFinder;
import org.fest.swing.fixture.FrameFixture;
import org.fest.swing.fixture.JOptionPaneFixture;
import org.jbehave.scenario.annotations.*;
import org.jbehave.scenario.steps.Steps;

import javax.swing.*;
import java.awt.*;
import java.util.List;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.notNullValue;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;


/**
 * User: tfdalt
 * Date: May 24, 2010
 * Time: 9:32:16 PM
 */
public class JbTrekSteps extends Steps {
    private String          userName = "testUser";
    private UserService     mockUserService;
    private FrameFixture    frameFixture;
    private Robot           robot;

    @BeforeScenario
    public void mockUserService() {
        mockUserService = mock(UserService.class);
    }

    @Given("user does not authenticate")
    public void userDoesNotAuthenticate() {
        when(mockUserService.isUserAuthenticated(userName)).thenReturn(false);
    }

    @Given("user does authenticate")
    public void userDoesAuthenticate() {
        when(mockUserService.isUserAuthenticated(userName)).thenReturn(true);
    }

    @Given("user fails retinal scan")
    public void userFailsRetinalScan() {
        when(mockUserService.retinaScanVerification(userName)).thenReturn(false);
    }

    @Given("user passes retinal scan")
    public void userPassesRetinalScan() {
        when(mockUserService.retinaScanVerification(userName)).thenReturn(true);
    }

    @Given("user has enablements $enablements")
    @Alias("user enablements are <enablements>")
    public void userHasEnablements(@Named("enablements") List<String> enablements) {
        for (Enablement enablement : Enablement.values()) {
            when(mockUserService.userHasEnablement(userName, enablement)).thenReturn(enablements.contains(enablement.name()));
        }
    }

    @When("the application is started")
    public void applicationStarted() {
        EventQueue.invokeLater(new Main(mockUserService, userName));
    }

    @Then("the message dialog '$message' is displayed")
    @When("the message dialog '$message' is displayed")
    public void messageDialogDisplayed(final String message) {
        JOptionPaneFixture optionPane = JOptionPaneFinder.findOptionPane(
            new GenericTypeMatcher<JOptionPane>(JOptionPane.class) {
                protected boolean isMatching(JOptionPane optionPane) {
                    return message.equals(optionPane.getMessage()) && optionPane.isShowing();
                }
            }).using(getRobot());
        assertThat(optionPane, notNullValue());
        optionPane.okButton().click();
    }

    @Then("the application is visible")
    public void applicationIsVisible() {
         frameFixture = WindowFinder.findFrame(
            new GenericTypeMatcher<JFrame>(JFrame.class) {
                protected boolean isMatching(JFrame frame) {
                    return "JBTrek".equals(frame.getTitle()) && frame.isShowing();
                }
            }).using(getRobot());
         assertThat(frameFixture, notNullValue());
    }

    @Then("the button with label '$label' is enabled")
    public void labelledButtonEnabled(String label) {
        frameFixture.button(labelled(label)).requireEnabled();  
    }

    @Then("the button with label '$label' is disabled")
    public void labelledButtonDisabled(String label) {
        frameFixture.button(labelled(label)).requireDisabled();
    }

    @Then("buttons labelled <enabledButtons> are enabled")
    public void buttonsEnabled(@Named("enabledButtons")  List<String> labels) {
        for (String label:labels) {
            labelledButtonEnabled(label);
        }
    }

    @Then("buttons labelled <disabledButtons> are disabled")
    public void buttonsDisabled(@Named("disabledButtons") List<String> labels) {
        for (String label:labels) {
            labelledButtonDisabled(label);
        }
    }

    private GenericTypeMatcher<JButton> labelled(final String label) {
        return new GenericTypeMatcher<JButton>(JButton.class) {

            @Override
            protected boolean isMatching(JButton component) {
                return component.getText().equals(label);
            }
        };
    }

    @AfterScenario
    public void closeFrameFixture() {
        if (frameFixture != null) {
            frameFixture.cleanUp();
        }
    }

    private Robot getRobot() {
        if (robot == null) {
            robot = BasicRobot.robotWithCurrentAwtHierarchy();
        }
        return robot;
    }
}
