package com.ociweb.mongoDB.most;

/*
 * Copyright (c) 2011, Object Computing, Inc.
 * All Rights reserved
 * See the file license.txt for licensing information.
 */
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.logging.Logger;

import org.bson.types.ObjectId;

import com.mongodb.BasicDBList;
import com.mongodb.BasicDBObject;
import com.mongodb.DBCollection;
import com.mongodb.DBCursor;
import com.mongodb.DBObject;
import com.mongodb.Mongo;
import com.sun.org.apache.bcel.internal.generic.NEW;

/**
 * 
 * simple test record which can be made to a specific size for testing.
 * 
 * @author Nathan Tippy
 * 
 */
public class TestRecord extends BasicDBObject {

    static Logger              logger      = SimpleHandler.simpleLoggger("com.ociweb.mongoDB.test.TestRecord");

    public static String       COUNT_KEY   = "c";
    public static String       GROUP_KEY   = "g";
    public static String       DATE_KEY    = "d";
    public static String       MESSAGE_KEY = "m";
    public static String       LIST_KEY    = "l";
    public static String       BINARY_KEY  = "b";

    // this constant should not be trusted if index, key names or record test
    // size changes!!
    private static int         MIN_SIZE    = 209; // WARNING DANGER magic number

    public TestRecord(Integer groupId, Long count, Date date, String message, List<Long> list, byte[] binary) {
        put(COUNT_KEY, count);
        put(GROUP_KEY, groupId);
        put(DATE_KEY, date);
        put(MESSAGE_KEY, message);
        put(LIST_KEY, list);
        put(BINARY_KEY, binary);
    }

    public static TestRecordBatchIterator mockIterator(final long documentSize, final long documentCount, final int batchSize, final Integer mask) throws InterruptedException {
        final String commonMessage = "message";

        final List<Long> commonList = new ArrayList<Long>();
        int i = 10;
        while (--i >= 0) {
            commonList.add((long) i);
        }

        i = (int) documentSize - MIN_SIZE;
        final byte[] commonBinary = new byte[i];
        while (--i >= 0) {
            commonBinary[i] = (byte) (i & 0xFF);
        }

        final Date commonDate = new Date();

        final AtomicLong offset = new AtomicLong();

        final int localMask = (mask == null ? 0xFFFFFFFF : mask);

        return new TestRecordBatchIterator() {

            @Override
            public DBObject[] getInitialBatch() {
                long myOffset = offset.getAndAdd(batchSize);
                TestRecord[] data = new TestRecord[batchSize];

                long count = myOffset;
                int b = 0;
                while (b < batchSize) {
                    data[b++] = new TestRecord((int) (count & localMask), count++, commonDate, commonMessage, commonList, commonBinary);
                }
                return data;
            }

            @Override
            public long updateToNextBatch(DBObject[] target) {
                long myOffset = offset.getAndAdd(batchSize);
                long count = myOffset;
                int limit = target.length;
                int b = 0;
                while (b < limit) {
                    target[b++] = new TestRecord((int) (count & localMask), count++, commonDate, commonMessage, commonList, commonBinary);
                }
                return myOffset;
            }

            @Override
            public long count() {
                return documentCount;
            }
        };
    }

    /**
     * Simple remote test data fetch implementation. It assumes that nothing
     * will be modifying the count of the remote collection documents while the
     * test runs. Its only suitable for running these load tests when the remote
     * server is known to have no other active users.
     * 
     * @param sourceDataMongo
     * @param testDataRemoteDatabaseName
     * @param testDataRemoteCollectionName
     * @return
     */
    public static TestRecordBatchIterator remoteIterator(final Mongo sourceDataMongo, final String testDataRemoteDatabaseName, final String testDataRemoteCollectionName, final int batchSize) {

        logger.info("Pulling remote data from " + sourceDataMongo.getAllAddress() + " " + testDataRemoteDatabaseName + "." + testDataRemoteCollectionName);

        return new TestRecordBatchIterator() {
            AtomicLong   offset     = new AtomicLong();

            DBCollection collection = sourceDataMongo.getDB(testDataRemoteDatabaseName).getCollection(testDataRemoteCollectionName);

            @Override
            public DBObject[] getInitialBatch() {
                int myOffset = (int) offset.getAndAdd(batchSize);
                DBObject[] data = new DBObject[batchSize];

                // find all in the natural order
                DBCursor cursor = collection.find().skip(myOffset).limit(batchSize);
                int b = 0;
                while (cursor.hasNext()) {
                    data[b++] = cursor.next();
                }
                cursor.close();
                return data;
            }

            @Override
            public long updateToNextBatch(DBObject[] data) {
                int myOffset = (int) offset.getAndAdd(batchSize);
                // find all in the natural order
                DBCursor cursor = collection.find().skip(myOffset).limit(batchSize);
                int b = 0;
                while (cursor.hasNext()) {
                    data[b++] = cursor.next();
                }
                cursor.close();
                return myOffset;
            }

            @Override
            public long count() {
                return collection.find().count();
            }
        };
    }
}
