package com.ociweb.mongoDB.most;

/*
 * Copyright (c) 2011, Object Computing, Inc.
 * All Rights reserved
 * See the file license.txt for licensing information.
 */
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Properties;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.bson.types.BSONTimestamp;
import org.bson.types.ObjectId;

import com.mongodb.BasicDBObject;
import com.mongodb.DB;
import com.mongodb.DBCollection;
import com.mongodb.DBCursor;
import com.mongodb.DBObject;
import com.mongodb.Mongo;
import com.mongodb.MongoException;
import com.mongodb.MongoOptions;
import com.mongodb.ServerAddress;
import com.mongodb.WriteConcern;

/**
 * 
 * MongoDB Stress Test application
 * 
 * @author Nathan Tippy
 * 
 */
public class MOST {

    Logger logger    = SimpleHandler.simpleLoggger("com.ociweb.mongoDB.test.MOST");
    String progTitle = " 0% _ _ _ _ _ _ _ _ _ _ _ _ 25% _ _ _ _ _ _ _ _ _ _ _ _ _ 50% _ _ _ _ _ _ _ _ _ _ _ _ _ 75% _ _ _ _ _ _ _ _ _ _ _ _ 100%";

    public enum Config {
        single, replica, shard
    };

    MongoDBShardedClusterServerFactory serverFactory;
    Long                               cappedSize;                                      // null
                                                                                        // is
                                                                                        // not
                                                                                        // capped
    String                             arguments                = "";
    String[]                           databaseFolders;
    Config                             configuration;

    String                             databaseName;
    String                             collectionName;
    boolean                            deleteOldData            = true;

    String[]                           shardConfigFolders;
    Integer                            shardChunkSize;                                  // null
                                                                                        // use
                                                                                        // the
                                                                                        // default
    int                                shardReplicaCount;
    Integer                            shardSplitMask;                                  // null
                                                                                        // use
                                                                                        // auto
                                                                                        // shard
    String                             shardKey                 = TestRecord.GROUP_KEY;

    Integer                            replicaSlaveDelay;

    long                               testDataMockDocumentSize;                       // bytes
    long                               testDataMockDocumentCount;

    boolean                            testNeverMasterOnSlaveOK = false;

    String[]                           testDataRemoteReplica;
    String                             testDataRemoteSingle;
    String                             testDataRemoteDatabaseName;
    String                             testDataRemoteCollectionName;

    int                                testBatchSize;
    int                                testThreads;
    WriteConcern                       testWriteConcern;
    boolean                            testFlushBeforeRead;
    boolean                            skipTest;
    boolean                            shutdownServersOnExit;
    boolean                            testSlaveOK;
    Level                              loggingLevel             = Level.INFO;

    public MOST(Properties mostProp) {

        loggingLevel = Level.parse(mostProp.getProperty("most.logLevel", "INFO"));
        logger.setLevel(loggingLevel);

        String installFolder = mostProp.getProperty("mongodb.install");
        
        // default here is 100 above mongos default just in case another instance is running
        int basePort = Integer.parseInt(mostProp.getProperty("mongodb.basePort", "27117"));

        String bindAddr = mostProp.getProperty("mongodb.bindAddr", "127.0.0.1");

        serverFactory = new MongoDBShardedClusterServerFactory(installFolder, bindAddr, basePort);

        MongoOptions mOptions = serverFactory.getMongoOptions();
        mOptions.autoConnectRetry = Boolean.parseBoolean(mostProp.getProperty("mongodb.options.autoConnectRetry", Boolean.toString(mOptions.autoConnectRetry)));
        mOptions.connectionsPerHost = Integer.parseInt(mostProp.getProperty("mongodb.options.connectionsPerHost", Integer.toString(mOptions.connectionsPerHost)));
        mOptions.connectTimeout = Integer.parseInt(mostProp.getProperty("mongodb.options.connectTimeout", Integer.toString(mOptions.connectTimeout)));
        mOptions.maxWaitTime = Integer.parseInt(mostProp.getProperty("mongodb.options.maxWaitTime", Integer.toString(mOptions.maxWaitTime)));
        mOptions.socketKeepAlive = Boolean.parseBoolean(mostProp.getProperty("mongodb.options.socketKeepAlive", Boolean.toString(mOptions.socketKeepAlive)));
        mOptions.socketTimeout = Integer.parseInt(mostProp.getProperty("mongodb.options.socketTimeout", Integer.toString(mOptions.socketTimeout)));
        mOptions.threadsAllowedToBlockForConnectionMultiplier = Integer.parseInt(mostProp.getProperty("mongodb.options.threadsAllowedToBlockForConnectionMultiplier", Integer.toString(mOptions.threadsAllowedToBlockForConnectionMultiplier)));

        arguments = mostProp.getProperty("mongodb.arguments", "");

        databaseFolders = mostProp.getProperty("mongodb.databaseFolders", "").split(";");

        // Values: single, replica, shard
        configuration = Enum.valueOf(Config.class, mostProp.getProperty("mongodb.configuration", "single"));

        databaseName = mostProp.getProperty("mongodb.dbName", "MOSTDB");
        collectionName = mostProp.getProperty("mongodb.collectionName", "test." + configuration.name());

        // when set to false data/configuration found in the existing folders is
        // used as is.
        deleteOldData = Boolean.parseBoolean(mostProp.getProperty("most.deleteOldData", "true"));

        // delimited by ;
        shardConfigFolders = mostProp.getProperty("mongodb.shard.configFolders").split(";");
        shardChunkSize = parseInteger(mostProp.getProperty("mongodb.shard.chunkSize", null));

        // 1 is the same as using single instances for each shard instead of
        // replica sets
        shardReplicaCount = Integer.parseInt(mostProp.getProperty("mongodb.shard.replicas", "1"));

        replicaSlaveDelay = parseInteger(mostProp.getProperty("mongodb.replica.slaveDelay", null));

        // without split mod it will do auto shard
        shardSplitMask = parseInteger(mostProp.getProperty("mongodb.shard.splitMask", null));
        // assert this value is all ones!

        testDataMockDocumentSize = Integer.parseInt(mostProp.getProperty("test.data.mock.documentSize", "4096"));
        testDataMockDocumentCount = Integer.parseInt(mostProp.getProperty("test.data.mock.documentCount", "1000000"));

        // Simicolon separated list of ip:port for source data
        testDataRemoteReplica = mostProp.getProperty("test.data.remote.replica", "").split(";");
        // Single ip:port for source data
        testDataRemoteSingle = mostProp.getProperty("test.data.remote.single", "");

        testDataRemoteDatabaseName = mostProp.getProperty("test.data.remote.databaseName", databaseName);
        testDataRemoteCollectionName = mostProp.getProperty("test.data.remote.collectionName", collectionName);

        // capped size is determined by the test document size and count
        // if its too small records will be lost as old ones are removed for new
        // inserts!!!
        float fudgeFactor = 1.02f;
        cappedSize = (Boolean.parseBoolean(mostProp.getProperty("mongodb.cappedCollection", "false")) ? (long) (testDataMockDocumentSize * testDataMockDocumentCount * fudgeFactor) : null);

        // cappedPct = 100; try running with smaller

        testFlushBeforeRead = Boolean.parseBoolean(mostProp.getProperty("test.flushBeforeRead", "false"));

        // use on write but only used by read when shard split mask is unset.
        // by default uses one thread for each core
        testThreads = Integer.parseInt(mostProp.getProperty("test.threads", Integer.toString(Runtime.getRuntime().availableProcessors())));
        logger.info("using " + testThreads + " test threads");

        testBatchSize = Integer.parseInt(mostProp.getProperty("test.batchSize", "100"));
        String stringWC = mostProp.getProperty("test.writeConcern", "NORMAL");

        // if this replicasCount ends up larger than the real number of replica
        // members the first writes will hang, with no notice,
        // and will wait for new memberes to be added to the set before
        // continuing. (do not let this happen, its hard to debug)
        int replicasCount = 0;
        switch (configuration) {
        case single:
            replicasCount = 0;
            break;
        case replica:
            replicasCount = databaseFolders.length;
            break;
        case shard:
            replicasCount = shardReplicaCount;
            if (databaseFolders.length % shardReplicaCount != 0) {
                logger.severe("The mongodb.shard.replicas must go evenly into the number of database folders when sharding!");
                System.exit(-1);
            }
            break;
        }

        testWriteConcern = ("REPLICAS_SAFE_ALL".equals(stringWC) ? new WriteConcern(replicasCount) : WriteConcern.valueOf(stringWC));

        testSlaveOK = Boolean.parseBoolean(mostProp.getProperty("test.slaveOK", "false"));
        testNeverMasterOnSlaveOK = !Boolean.parseBoolean(mostProp.getProperty("test.masterOKslaveOK", "false"));

        skipTest = Boolean.parseBoolean(mostProp.getProperty("most.skipTest", "false"));
        shutdownServersOnExit = Boolean.parseBoolean(mostProp.getProperty("most.shutdownServersOnExit", "true"));

        Logger.getLogger("com.ociweb.mongoDB.most.MongoDBServerFactory").setLevel(loggingLevel);
        Logger.getLogger("com.ociweb.mongoDB.test.TestRecord").setLevel(loggingLevel);

        // print the details of what we are about to test.
        logger.info("Config:" + configuration + "  WriteConcern:" + stringWC + (arguments.length() > 0 ? " Arguments:" + arguments : "") + (cappedSize != null ? " Capped:" + cappedSize : ""));
        if (!shutdownServersOnExit) {
            logger.info("Will NOT shut down servers upon exit");
        }
        switch (configuration) {
        case single:
            break;
        case replica:
            logger.info("slaveOK:" + testSlaveOK + "  masterOKslaveOK:" + (!testNeverMasterOnSlaveOK));
            break;
        case shard:
            if (replicasCount > 1) {
                logger.info("slaveOK:" + testSlaveOK + "  masterOKslaveOK:" + (!testNeverMasterOnSlaveOK));
            }
            break;
        }

    }

    private Integer parseInteger(String value) {
        return (value == null ? null : Integer.parseInt(value));
    }

    public Mongo buildMongo() throws UnknownHostException, InterruptedException {

        int maxCon = serverFactory.getMongoOptions().connectionsPerHost;
        int confCon = (shardSplitMask == null ? testThreads : Math.max(testThreads, shardSplitMask));

        if (confCon > maxCon) {
            logger.warning("Driver will only open " + maxCon + " connections but this configuration will use " + confCon);
        }

        Mongo mongo = null;
        switch (configuration) {
        case single:
            mongo = serverFactory.startup(databaseFolders[0], arguments, cappedSize, databaseName, collectionName, deleteOldData);
            break;
        case replica:
            mongo = serverFactory.startup(databaseFolders, arguments, cappedSize, replicaSlaveDelay, databaseName, collectionName, deleteOldData);
            break;
        case shard:

            int reqCons = serverFactory.mongosRequiredConnectionsCount(databaseFolders.length / shardReplicaCount, databaseFolders.length, confCon);

            logger.info("mongos will require at most " + reqCons + " connections.");

            mongo = serverFactory.startup(databaseFolders, shardConfigFolders, arguments, shardReplicaCount, replicaSlaveDelay, shardChunkSize, shardSplitMask, shardKey, databaseName, collectionName, deleteOldData);
            break;
        }
        // Must NEVER set slaveOK here on mongo because it can not be turned off
        // and will impact all
        // the admin commands required to read the size of the data written
        return mongo;
    }

    public static void main(String[] arg) throws InterruptedException, IOException {

        Properties mostProp = loadProperties(arg);
        if (mostProp != null) {
            // we now have all the properties, start the app
            MOST mongoStressTest = new MOST(mostProp);
            // build the test configuration of servers
            Mongo mongo = mongoStressTest.buildMongo();
            // run the full load test
            mongoStressTest.runSimpleRecordTest(mongo);
            // shutdown the servers if enabled
            mongoStressTest.shutdown(mongo);
        }

    }

    private static Properties loadProperties(String[] arg) throws IOException {
        InputStream in;

        if (arg.length > 0 && (new File(arg[0]).isFile())) {
            in = new FileInputStream(arg[0]);
        } else {
            String propFile = "/most.properties";
            in = MOST.class.getResourceAsStream(propFile);
            if (in == null) {
                // failure before we have the logger instance
                System.err.println("unable to find " + propFile);
                return null;
            }
        }

        Properties mostProp = new Properties();
        try {
            mostProp.load(in);
        } catch (Exception e) {
            // failure before we have the logger instance
            e.printStackTrace();
            return null;
        }
        in.close();
        return mostProp;
    }

    public void shutdown(Mongo mongo) {

        // close our test mongo instance
        mongo.close();

        // request clean shutdown of the servers
        if (shutdownServersOnExit) {
            serverFactory.shutdownServers();
        }
    }

    public void runSimpleRecordTest(Mongo mongo) throws InterruptedException, UnknownHostException, MongoException {
        if (skipTest) {
            return; // do nothing in this method
        }

        DB db = mongo.getDB(databaseName);
        DBCollection collection = db.getCollection(collectionName);

        // just to prove we are not cheating.
        long initialCollectionSizeInBytes = ((Number) collection.getStats().get("size")).longValue();
        if (initialCollectionSizeInBytes > 0) {
            logger.severe("Collection must be empty before starting test found:" + initialCollectionSizeInBytes);
            return;
        }

        
        long begin;
        TestRecordBatchIterator dataIterator = null;
        if (testDataRemoteSingle.length() > 0) {
        	shardSplitMask = null;//using remote source so we can not query on this
        	logger.info("Single source: "+testDataRemoteSingle);
            Mongo sourceDataMongo = new Mongo(testDataRemoteSingle, serverFactory.getMongoOptions());
            DBCollection col = sourceDataMongo.getDB(testDataRemoteDatabaseName).getCollection(testDataRemoteCollectionName);
            testDataMockDocumentCount = col.find().count();

            // simple test with remote data (may saturate network connection)
            dataIterator = TestRecord.remoteIterator(sourceDataMongo, testDataRemoteDatabaseName, testDataRemoteCollectionName, testBatchSize);

        } else if (testDataRemoteReplica[0].length() > 0) {
        	shardSplitMask = null;//using remote source so we can not query on this
            List<ServerAddress> replicaSetSeeds = new ArrayList<ServerAddress>();
            for (String single : testDataRemoteReplica) {
                String[] hostPort = single.split(":");
                replicaSetSeeds.add(new ServerAddress(hostPort[0], Integer.parseInt(hostPort[1])));
            }
            logger.info("Replica set source: "+replicaSetSeeds);
            Mongo sourceDataMongo = new Mongo(replicaSetSeeds, serverFactory.getMongoOptions());
            DBCollection col = sourceDataMongo.getDB(testDataRemoteDatabaseName).getCollection(testDataRemoteCollectionName);
            testDataMockDocumentCount = col.find().count();

            if (testDataMockDocumentCount == 0) {
                logger.severe("unable to find any records for " + testDataRemoteDatabaseName + " and " + testDataRemoteCollectionName);
                serverFactory.shutdownServers();
                System.exit(-1);
            }
            // simple test with remote data (may saturate network connection)
            dataIterator = TestRecord.remoteIterator(sourceDataMongo, testDataRemoteDatabaseName, testDataRemoteCollectionName, testBatchSize);
        } else {
            // simple test with mock data
            dataIterator = TestRecord.mockIterator(testDataMockDocumentSize, testDataMockDocumentCount, testBatchSize, shardSplitMask);
        }

        begin = System.currentTimeMillis();
        runSimpleWriteTest(collection, dataIterator);
        long endTime = System.currentTimeMillis();

        long writeDuration = endTime - begin;
        // in the rare case that its below the measurement threshold we still
        // need to divide
        if (writeDuration == 0) {
            writeDuration = 1;
        }

        long insertsPerSecond = 1000l * testDataMockDocumentCount / writeDuration;

        logger.info("wrote " + testDataMockDocumentCount + " docs at " + insertsPerSecond + " docs/sec, duration " + (writeDuration / 1000) + " sec");

        if (testFlushBeforeRead) {
            logger.info("Flushing all writes before beginning read test.");
            // NOTE:May need to run test once first to ensure data fiels are
            // allocated.
            long startFSync = System.currentTimeMillis();
            // must wait for in memory flush to finish before checking.

            logger.info("mongo> use admin");
            int serverCount = mongo.getAllAddress().size();
            BasicDBObject cmd = new BasicDBObject("fsync", serverCount).append("async", "false");

            logger.info("mongo> db.runCommand(" + cmd + ")");
            logResult(mongo.getDB("admin").command(cmd));

            cmd = new BasicDBObject("getLastError", 1).append("w", serverCount);
            logResult(mongo.getDB("admin").command(cmd));

            logger.info("fsync duration: " + (System.currentTimeMillis() - startFSync) + " ms, collection stats:" + collection.getStats());
            if (testSlaveOK) {
                long syncStart = System.currentTimeMillis();
                int seconds = outOfSyncSeconds(mongo);
                if (seconds > 0) {
                    logger.info("Replica sets are out of sync by:" + seconds + " waiting for replication to complete...");
                    while (outOfSyncSeconds(mongo) > 0) {
                        Thread.yield();
                    }
                    long syncDuration = System.currentTimeMillis() - syncStart;
                    logger.info("replication duration: " + syncDuration + " ms");
                }
            }
        }

        // start read timer
        begin = System.currentTimeMillis();
        long found = runSimpleReadTest(collection, testDataMockDocumentCount, (shardSplitMask == null ? 1 : shardSplitMask));

        long readDuration = System.currentTimeMillis() - begin;
        // in the rare case that its below the measurement threshold we still
        // need to divide
        if (readDuration == 0) {
            readDuration = 1;
        }

        long finalCollectionSizeInBytes = ((Number) collection.getStats().get("size")).longValue();

        // testing to help clean error
        int serverCount = mongo.getAllAddress().size();
        DBObject cmd = new BasicDBObject("getLastError", 1).append("w", serverCount);
        logResult(mongo.getDB("admin").command(cmd));

        assert (readDuration > 0);
        assert (finalCollectionSizeInBytes > 0);
        assert (found > 0);
        assert (testDataMockDocumentCount > 0);
        assert (found <= testDataMockDocumentCount);
        float findsPerSecond = 1000l * found / readDuration;

        String writeHumanBPS = humanBPS(testDataMockDocumentCount, writeDuration, finalCollectionSizeInBytes);
        logger.info("bytes written " + finalCollectionSizeInBytes + " " + writeHumanBPS + " written");

        String readHumanBPS = humanBPS(found, readDuration, finalCollectionSizeInBytes);

     //   logger.severe("found: " + found + " count " + testDataMockDocumentCount);
     //   logger.severe(collection.getStats().toString());

        long pctFound = 100l * found / testDataMockDocumentCount;
        int notReadableYet = (int) (testDataMockDocumentCount - found);

        logger.info("read finished, " + findsPerSecond + " docs/sec, " + readHumanBPS + " found " + pctFound + "% " + (notReadableYet > 0 ? "(" + notReadableYet + " not written before find)" : "") + " duration " + (readDuration / 1000) + " sec");

    }

    private String humanBPS(long docCount, long duration, long finalCollectionSizeInBytes) {
        int bytesPerSecond = (int) (((finalCollectionSizeInBytes / testDataMockDocumentCount) * docCount * 1000d) / duration);
        String humanBPS = (bytesPerSecond >> 20 == 0 ? (bytesPerSecond >> 10) + " KB/sec" : (bytesPerSecond >> 20) + " MB/sec");
        return humanBPS;
    }

    /**
     * If the command result indicates that no error occurred then it will be
     * logged as fine. Otherwise its logged as a severe error and the
     * application exits.
     * 
     * @param result
     * @return
     */
    protected DBObject logResult(DBObject result) {
        if (((Number) result.get("ok")).intValue() == 1) {
            logger.fine(result.toString());
        } else {
            logger.severe(result.toString());
            System.exit(-1);
        }
        return result;
    }

    private void runSimpleWriteTest(final DBCollection collection, final TestRecordBatchIterator dataIterator) throws InterruptedException {

        ExecutorService pool = Executors.newFixedThreadPool(testThreads);
        final long documentCount = dataIterator.count();

        logger.info("Waiting for writes to complete.");
        final boolean showProgressBar = loggingLevel.intValue() >= MongoDBBaseServerFactory.instanceLogLevel.intValue();
        if (showProgressBar) {
            System.out.println(progTitle);
        }
        final Lock progLock = new ReentrantLock();
        final AtomicInteger progWritten = new AtomicInteger();
        final Object insertLock = new Object();

        // loop for each thread
        int c = testThreads;
        while (--c >= 0) {
            pool.execute(new Runnable() {

                long factor = documentCount / 120l;

                @Override
                public void run() {
                    try {
                        DBObject[] batch = dataIterator.getInitialBatch();
                        long count = 0;
                        long stop = documentCount - batch.length;

                        do {
                            if (count > stop) {
                                batch = Arrays.copyOfRange(batch, 0, (int) (documentCount - count));
                            }
                            
                            
                            //This syncronized block is here for a number of reasons.
                            
                            // - it forces a wait so we can measure the difference 
                            //   between NORMAL and SAFE
                            
                            // - it protects from a bug in the Java driver relating
                            //   to sharded multi-threaded bulk inserts
                            
                            // - it helps speed up the read time.  Because more of 
                            //   the writes are done in series the reads will be done
                            //   with less search head motion.
                            
                            
                            synchronized (insertLock) {
                                collection.insert(batch, testWriteConcern);
                            }

                            if (showProgressBar && progLock.tryLock()) {
                                try {
                                    int p = progWritten.get();
                                    while (p * factor < count) {
                                        System.out.print('=');
                                        System.out.flush();
                                        p = progWritten.incrementAndGet();
                                    }
                                } finally {
                                    progLock.unlock();
                                }
                            }

                        } while ((count = dataIterator.updateToNextBatch(batch)) < documentCount);
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                }
            });
        }

        pool.shutdown();
        pool.awaitTermination(10, TimeUnit.HOURS);
        System.out.println();
    }

    /**
     * Multi threaded reading test. Does each group in parallel and further
     * divides each group in to smaller parts
     * 
     * @param collection
     * @param totalCount
     * @param groups
     *            same as the sharding key
     * @param parts
     *            threads to be used per group
     * @return
     * @throws InterruptedException
     */
    private int runSimpleReadTest(final DBCollection collection, final long totalCount, final int shardMask) throws InterruptedException {

        final int groupCount = (shardMask > 1 ? shardMask + 1 : 1);

        final long factor = testDataMockDocumentCount / 120l;

        logger.info("Waiting for reads to complete.");
        final boolean showProgressBar = loggingLevel.intValue() >= MongoDBBaseServerFactory.instanceLogLevel.intValue();
        if (showProgressBar) {
            System.out.println(progTitle);
        }
        final Lock projLock = new ReentrantLock();
        final AtomicInteger progWritten = new AtomicInteger();

        ExecutorService pool = Executors.newFixedThreadPool(groupCount > 1 ? groupCount : testThreads);
        final AtomicInteger count = new AtomicInteger();

        // serverCount is used to split parts smaller than the count of threads
        // to ensure a better balance in replica sets.
        // there is no guarantee that the java driver will round robin so making
        // more smaller requests helps cover this issue.
        int serverCount = collection.getDB().getMongo().getAllAddress().size();
        int masterMod = 0;
        // only split by threads if mod is not greater than one
        long part = 1 + (totalCount / ((groupCount > 1 ? 1 : testThreads) * serverCount));// added
                                                                                          // 1
                                                                                          // to
                                                                                          // round
                                                                                          // up
                                                                                          // fractions

        long start = 0;
        while (start < totalCount) {
            // when no mask is used groups must be 1 so we don't loop multiple times
            int groups = groupCount;
            while (--groups >= 0) {
                final int g = groups;
                final long gte = start;
                final long lt = start + part;

                final int server = (masterMod % serverCount);

                pool.execute(new Runnable() {

                    @Override
                    public void run() {

                        BasicDBObject query = new BasicDBObject(TestRecord.COUNT_KEY, new BasicDBObject("$gte", gte).append("$lt", lt));
                        if (groupCount > 1) {
                            if (lt == totalCount) {
                                query = new BasicDBObject(TestRecord.GROUP_KEY, g);
                            } else {
                                query = query.append(TestRecord.GROUP_KEY, g);
                            }
                        }
                        // Descending order so we might be able to get this read
                        // in before the write to disk was completed
                        DBCursor cursor = collection.find(query).sort(new BasicDBObject(TestRecord.COUNT_KEY, -1));

                        // when neverMasterOnSlaveOK is false ever Nth server is
                        // forced to be master by not setting slaveOK.
                        if (testSlaveOK && ((server != 0) || testNeverMasterOnSlaveOK)) {
                            if (collection.getDB().getMongo().getAllAddress().size() == 1) {
                                logger.finer("SlaveOK is true but configuration is only connected to a single instance. port:" + collection.getDB().getMongo().getAllAddress().get(0).getPort());
                            }
                            cursor = cursor.slaveOk();
                        }

                        try {
                            boolean showFinerDetails = (logger.getLevel().intValue() <= Level.FINER.intValue());
                            int localCount = 0;

                            ObjectId lastId = new ObjectId();
                            while (cursor.hasNext()) {
                                DBObject document = cursor.next();

                                // could be done as snapshot however this is not
                                // supported with the use of sort
                                ObjectId id = (ObjectId) document.get("_id");
                                if (id.equals(lastId)) {
                                    logger.finer("duplicates skipped " + document.toString());
                                    continue; // skip any duplicates we may have because we are not using snapshot
                                }
                                lastId = id;

                                // count all the read documents in groups of 256
                                if (((++localCount) & 0xFF) == 0) {
                                    assert (localCount == 256) : "localCount " + localCount;
                                    int value = count.addAndGet(256);
                                    localCount = 0;

                                    if (showFinerDetails) {
                                        logger.severe("read test query: " + query + " count:" + cursor.count() + " " + cursor.getServerAddress());
                                        showFinerDetails = false;
                                    }

                                    if (showProgressBar && (projLock.tryLock() || value == testDataMockDocumentCount)) {
                                        try {
                                            int p = progWritten.get();
                                            while (p * factor < value) {
                                                System.out.print('=');
                                                System.out.flush();
                                                p = progWritten.incrementAndGet();
                                            }
                                        } finally {
                                            projLock.unlock();
                                        }
                                    }
                                }
                            }
                            count.addAndGet(localCount);
                        } catch (Exception e) {
                            e.printStackTrace();
                        } finally {
                            cursor.close();
                        }
                    }
                });
            }
            start = start + part;
            masterMod++;
        }
        pool.shutdown();
        pool.awaitTermination(10, TimeUnit.HOURS);
        System.out.println();
        return count.get();

    }

    /**
     * determine if any slave in replSet is out of sync
     */
    public int outOfSyncSeconds(Mongo mongo) {

        DBObject result;
        try {
            result = mongo.getDB("admin").command(new BasicDBObject("replSetGetStatus", 1));
        } catch (Exception e) {
            // network trouble so this can not be measured right now
            // assume that we are greatly out of sync
            return Integer.MAX_VALUE;
        }
        List<DBObject> members = (List<DBObject>) result.get("members");
        if (members == null) {
            return 0;
        }

        // remove all the "RECOVERING" members they will not be used as part of
        // slaveOk
        int lastIndex = members.size();
        while (--lastIndex >= 0) {
            if ("RECOVERING".equals(members.get(lastIndex).get("stateStr"))) {
                members.remove(lastIndex);
            }

        }

        // Count everything that is not RECOVERING
        lastIndex = members.size() - 1;
        BSONTimestamp opTime = (BSONTimestamp) members.get(lastIndex).get("optime");

        // compute largest spread between between primary and the slaves
        int minSec = opTime.getTime();
        int maxSec = opTime.getTime();

        int i = lastIndex;
        while (--i >= 0) {

            BSONTimestamp temp = (BSONTimestamp) members.get(i).get("optime");
            int seconds = temp.getTime();
            if (seconds > maxSec) {
                maxSec = seconds;
            }
            if (seconds < minSec) {
                minSec = seconds;
            }
        }

        return maxSec - minSec;
    }

}
