package com.ociweb.mongoDB.most;

/*
 * Copyright (c) 2011, Object Computing, Inc.
 * All Rights reserved
 * See the file license.txt for licensing information.
 */
import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.logging.Level;
import java.util.logging.Logger;

import com.mongodb.BasicDBObject;
import com.mongodb.DB;
import com.mongodb.DBCollection;
import com.mongodb.DBObject;
import com.mongodb.Mongo;
import com.mongodb.MongoOptions;

/**
 * 
 * Base class for all three server launching factories.
 * 
 * @author Nathan Tippy
 * 
 */
public abstract class MongoDBBaseServerFactory {

    public static Level  instanceLogLevel = Level.FINER;

    Logger               logger           = SimpleHandler.simpleLoggger("com.ociweb.mongoDB.most.MongoDBServerFactory");
    String               separator        = System.getProperty("file.separator");

    String               mongoDBPath;
    final int            basePort;               // first
                                                 // port
                                                 // to
                                                 // be
                                                 // used
                                                 // by
                                                 // the
                                                 // first
                                                 // server
                                                 // instance
    int                  nextPort;       // next
                                         // port
                                         // to
                                         // be
                                         // used
                                         // by
                                         // the
                                         // next
                                         // server
                                         // instance
    final String         bindAddr;
    MongoOptions         mOptions;
    Map<Integer, Logger> instanceLoggers  = new HashMap<Integer, Logger>();

    public MongoDBBaseServerFactory(String mongoDBPath, String bindAddr, int basePort) {

        this.mongoDBPath = mongoDBPath;
        this.basePort = basePort;
        this.nextPort = basePort;
        this.bindAddr = bindAddr;

        // need to pass in each of these props
        mOptions = new MongoOptions();
        mOptions.description = "MOST";
    }

    public MongoOptions getMongoOptions() {
        return mOptions;
    }

    protected void launchMongoDBInstance(int port, String mongoDBStartupCommand, AtomicInteger serversUp) {

        // before launching this instance make sure that any from previous runs
        // are really shut down.
        shutdownSingleServerByPort(port);// must restore normal logging for
                                         // com.mongodb when done with this call
        Logger.getLogger("com.mongodb").setLevel(Level.SEVERE);

        logger.info("CMD> " + mongoDBStartupCommand);
        try {
            Process p = Runtime.getRuntime().exec(mongoDBStartupCommand);
            instanceLoggers.put(port, SimpleHandler.simpleLoggger(mongoDBStartupCommand));

            addStreamConsumer(port, mongoDBStartupCommand, instanceLogLevel, new BufferedReader(new InputStreamReader(p.getInputStream())), serversUp);
            addStreamConsumer(port, mongoDBStartupCommand, Level.SEVERE, new BufferedReader(new InputStreamReader(p.getErrorStream())), null);

        } catch (IOException e) {
            logger.severe(e.getMessage());
        }
    }

    /**
     * Write each line of test to the appropriate log level. If these are not
     * consumed quickly the servers may hang.
     */
    private void addStreamConsumer(final Integer port, final String command, final Level level, final BufferedReader input, final AtomicInteger serversUp) {
        Thread t = new Thread(new Runnable() {

            @Override
            public void run() {
                Logger myLogger = instanceLoggers.get(port);
                myLogger.setLevel(logger.getLevel());
                try {

                    String line;
                    int count = 0;
                    boolean isDown = true;
                    while ((line = input.readLine()) != null) {
                        count++;

                        // bump up the count of active servers if this one is
                        // waiting for connections
                        // NOTE: this will break if 10gen ever changes the
                        // output text
                        if (isDown && serversUp != null && line.contains("waiting for connections on port " + port)) {
                            serversUp.incrementAndGet();
                            isDown = false;
                        }

                        myLogger.log(level, bindAddr + ":" + port + " " + line);

                        ////This block is just a test that did not work while trying to shutdown the server.
                        
                        // if the server has crashed to the point that it needs
                        // to show a memory map we will need to
                        // attemp a kill before it will go away. Should not be
                        // needed but there is a bug in mongos
                        if (line.contains("======= Memory map: ========")) {
                            final Thread t = new Thread(new Runnable() {

                                @Override
                                public void run() {
                                    try {// SIGTERM
                                        Runtime.getRuntime().exec("pkill -15 mongos");
                                    } catch (IOException e) {
                                    }
                                    try {
                                        Thread.currentThread().wait(10000);
                                        // 10 seconds
                                    } catch (InterruptedException e1) {
                                    }

                                    try {// SIGKILL
                                        Runtime.getRuntime().exec("pkill -9 mongos");
                                    } catch (IOException e) {
                                    }

                                }
                            });
                            t.start();
                            // this better take less time than the above 10
                            // seconds because we are not willing to wait any
                            // longer
                        }
                        //end of attempted shutdown block
                        

                    }
                    input.close();

                    if ((count > 0) && (level == Level.SEVERE)) {
                        shutdownServers();// extra call to shut down the
                                          // servers, something has gone
                                          // wrong
                    }
                } catch (IOException e) {
                    myLogger.throwing("MongoDBServerFactory", "addConsumer", e);
                }
            }
        });
        // if these system outs are not taken away quickly they will stop up the
        // server
        t.setPriority(Thread.MAX_PRIORITY);
        t.start();
    }

    /**
     * Creates the target folder if it does not exits. If it does exit checks to
     * see if it was left with and old mongod.lock from a previous run which may
     * have been interrupted.
     * 
     * Deleting this lock file is never a good idea however because we are about
     * to delete all the data for the test anyway it does not matter. If fast
     * restarts are a requirement in your production environment the --journal
     * feature should be used.
     * 
     * @param dbFolder
     */
    protected void validateTargetFolderForMongoDB(String dbFolder, boolean deleteOldData) {
        if (dbFolder == null) {
            return;// nothing to check
        }
        File folder = new File(dbFolder);
        if (!folder.exists()) {
            if (!folder.mkdir()) {
                logger.severe("Unable to create required folder: " + dbFolder);
                shutdownServers();
                System.exit(-1);
            } else {
                logger.finer("created new folder: " + dbFolder);
            }
        } else {
            if (deleteOldData) {
                logger.finer("delete old data in folder: " + dbFolder);
                // need to delete the old files which might still be there from
                // a previous run.
                deleteFolder(folder);
            }
        }
    }

    /**
     * Recursively delete all the files in this folder and all child folders
     * 
     * @param folder
     */
    private void deleteFolder(File folder) {
        for (File file : folder.listFiles()) {
            if (file.isFile()) {
                boolean success = file.delete();
                if (!success) {
                    logger.severe("Unable to delete old file:" + file.getAbsolutePath());
                    shutdownServers();
                    System.exit(-1);
                }
            } else {
                deleteFolder(file);
            }
        }
    }

    /**
     * Waits until [serverCount] servers are up and ready to accept connections
     * 
     * @param serverCount
     * @throws InterruptedException
     */
    protected void awaitFor(AtomicInteger serversUp, int serverCount) {

        // there are better ways to do this but this is more clear and
        // this will never be used while the load tests are running.
        long startTime = System.currentTimeMillis();
        boolean displayedNotice = false;
        Level oldLevel = null;
        boolean restoreLevel = false;
        while (serversUp.get() < serverCount) {
            try {
                Thread.sleep(10);
                long duration = System.currentTimeMillis() - startTime;
                if (duration > 10000) {// if we have been waiting longer than 10
                                       // seconds print some details
                    int remaining = (serverCount - serversUp.get());
                    if (!displayedNotice) {
                        logger.info("Wating on " + remaining + " out of " + serverCount + " servers to accept connections.");

                        oldLevel = logger.getLevel();
                        if (oldLevel.intValue() < Level.INFO.intValue()) {
                            logger.setLevel(instanceLogLevel);// show all
                                                              // because we are
                                                              // getting
                                                              // worried.
                            for (Logger l : instanceLoggers.values()) {
                                l.setLevel(instanceLogLevel);
                            }
                            restoreLevel = true;
                        }

                        displayedNotice = true;
                    }
                    if (duration > (1200000)) { // 20 minute timeout
                        logger.severe("timeout waiting for " + remaining + " servers to accept connections");
                        logger.severe("please run again with log level set to fine to debug the servers");
                        shutdownServers();
                        System.exit(-1);
                    }

                }
            } catch (InterruptedException ie) {
                shutdownServers();
                System.exit(-1);
            }
        }
        if (restoreLevel) {
            logger.setLevel(oldLevel);
            for (Logger l : instanceLoggers.values()) {
                l.setLevel(oldLevel);
            }
        }
    }

    protected DBObject logResult(Mongo host, DBObject result, DBObject command) {
        return logResult(host, result, command, null);
    }

    /**
     * If the command result indicates that no error occurred then it will be
     * logged as fine. Otherwise its logged as a severe error and the
     * application exits.
     * 
     * @param result
     * @return
     */
    protected DBObject logResult(Mongo host, DBObject result, DBObject command, String ignoreMsg) {
        if ((((Number) result.get("ok")).intValue() == 1) || (ignoreMsg != null && result.get("errmsg").equals(ignoreMsg))) {
            logger.fine(host.debugString() + " result: " + result.toString() + " command: " + command);
        } else {
            logger.severe(host.debugString() + " result: " + result.toString() + " command: " + command);
            shutdownServers();
            System.exit(-1);
        }
        return result;
    }

    /**
     * Build collection needed for running the load test. The collection will be
     * capped if cappedSize is not null.
     * 
     * @param primary
     * @param cappedSize
     *            should be null in order to use normal collections.
     * @return
     */
    protected Mongo buildTestCollection(Mongo primary, Long cappedSize, String dbName, String collectionName) {
        // must erase old data first and setup shards
        DB db = primary.getDB(dbName);
        logger.info("mongo> use " + dbName);
        // build capped or un-capped collection
        DBCollection collection;
        if (cappedSize == null) {
            logger.info("mongo> db." + collectionName);
            collection = db.getCollection(collectionName);
        } else {
            DBObject cap = new BasicDBObject();
            cap.put("capped", true);
            cap.put("size", cappedSize);// in bytes
            logger.info("mongo> db.createCollection(" + collectionName + "," + cap + ")");
            collection = db.createCollection(collectionName, cap);
        }
        // add composite index to our collection to assist in reading the
        // results via query and to demonstrate that the index is built quickly
        // as records are inserted.
        collection.ensureIndex(new BasicDBObject(TestRecord.COUNT_KEY, 1).append(TestRecord.GROUP_KEY, 1).append("unique", "true"));
        collection.ensureIndex(new BasicDBObject(TestRecord.GROUP_KEY, 1));

        return primary;
    }

    /**
     * Shutdown all the servers in the reverse order that they were started.
     * 
     * @throws InterruptedException
     */
    public void shutdownServers() {

        ExecutorService pool = Executors.newFixedThreadPool(nextPort - basePort);
        int p = nextPort;
        while (--p >= basePort) {
            final int port = p;
            pool.execute(new Runnable() {
                public void run() {
                    shutdownSingleServerByPort(port);
                }
            });
        }
        pool.shutdown();
        try {
            pool.awaitTermination(10, TimeUnit.HOURS);
        } catch (InterruptedException e) {
            // exit
        }
        // this method is called at shutdown so the logging for com.mongodb is
        // not returned to the normal state
        // Logger.getLogger("com.mongodb").setLevel(Level.SEVERE);
        // un-commenting the line above may reveal some of the incorrect errors
        // thrown at shutdown
    }

    /**
     * Shutdown a single server running here on localhost (127.0.0.1).
     * 
     * @param port
     */
    private void shutdownSingleServerByPort(Integer port) {

        Mongo mongo = null;
        try {
            // localhost is hardcoded because this will not work with any other

            mongo = new Mongo("127.0.0.1:" + port);
            // must turn off debug message or driver will complain that the
            // server is no longer up.
            Logger.getLogger("com.mongodb").setLevel(Level.OFF);
            // this command only works on localhost connections or with
            // authentication
            DB db = mongo.getDB("admin");
            if (db != null && db.getStats() != null) {
                logger.info("requesting clean shut down of server on port: " + port);
                BasicDBObject cmd = new BasicDBObject("shutdown", 1);
                logResult(mongo, db.command(cmd), cmd);

            }
        } catch (Throwable e) {
            // do nothing, the server will be left running and will need to be
            // killed via the task manager
        } finally {
            try {
                mongo.close();
            } catch (Throwable e) {
                // eat this exception like the other above we are closing mongo
                // so we do not care
            }
        }
    }
}
