package com.ociweb.mongoDB.most;

/*
 * Copyright (c) 2011, Object Computing, Inc.
 * All Rights reserved
 * See the file license.txt for licensing information.
 */
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.atomic.AtomicInteger;

import com.mongodb.BasicDBObject;
import com.mongodb.DBObject;
import com.mongodb.Mongo;
import com.mongodb.MongoException;
import com.mongodb.ServerAddress;

/**
 * 
 * factory for launching replica set or single instance.
 * 
 * @author Nathan Tippy
 * 
 */
public class MongoDBReplicaSetServerFactory extends MongoDBSingleInstanceServerFactory {

    public MongoDBReplicaSetServerFactory(String mongoDBPath, String bindAddr, int basePort) {
        super(mongoDBPath, bindAddr, basePort);
    }

    protected List<ServerAddress> launchEmptyReplicaSet(String[] dbFolders, String setName, Integer slaveDelay, String arguments, boolean deleteOldData) {

        int memberCount = dbFolders.length;
        if ((memberCount & 1) != 1) {
            logger.warning("Replica sets work best with an odd number of members. Testing with:" + memberCount + " members");
        }

        List<ServerAddress> servers = new ArrayList<ServerAddress>();
        List<DBObject> memberList = new ArrayList<DBObject>();

        AtomicInteger serversUp = new AtomicInteger();
        int id = 0;

        for (String dbFolder : dbFolders) {
            String host = launchEmptySingle("mongod", dbFolder, "--replSet " + setName + " " + arguments, serversUp, deleteOldData);
            try {
                servers.add(new ServerAddress(host));
            } catch (UnknownHostException e) {
                logger.severe(e.getMessage());
                shutdownServers();
                System.exit(-1);
            }

            BasicDBObject singleConfig = new BasicDBObject("_id", id++).append("host", host);
            if (slaveDelay != null && memberList.size() > 0) {
                // all others will be slave delay
                singleConfig.append("priority", 0); // can only use slaveDelay
                                                    // if priority is zero.
                singleConfig.append("slaveDelay", slaveDelay); // in seconds
            }

            memberList.add(singleConfig);
        }

        // wait for all of the replica set members to be up and accepting
        // connections before we send the configuration
        awaitFor(serversUp, memberCount);

        if (deleteOldData) {
            try {
                // we will use the first member to establish the replica set,
                // any one
                // would work. Do not take basePort this may be something else
                // when used by shard
                Mongo initialMember = new Mongo((String) memberList.get(0).get("host"), mOptions);
                replSetInitiate(memberList, initialMember, setName);

                // wait for a PRIMARY to be elected from the set
                // no longer need connection initialMember we will now connect
                // to the set
                awaitElection(initialMember).close();
            } catch (Exception e) {
                logger.severe(e.getMessage());
                shutdownServers();
                System.exit(-1);
            }

        } else {
            awaitElection(new Mongo(servers)).close();

        }
        // The Mongo object will be aware of all the servers so it can
        // dynamically
        // switch to the newly elected primary whenever the current primary
        // should go down.
        return servers;
    }

    private void replSetInitiate(List<DBObject> memberList, Mongo initialMember, String setName) {
        // this JSON object defines all the members for the initial server
        DBObject replSetConfig = new BasicDBObject("_id", setName).append("members", memberList);
        BasicDBObject cmd = new BasicDBObject("replSetInitiate", replSetConfig);
        logger.info("mongo> use admin");
        logger.info("mongo> db.runCommand(" + cmd + ")");
        // send replSetInitiate command and exit it it fails.
        logResult(initialMember, initialMember.getDB("admin").command(cmd), cmd);
    }

    private Mongo awaitElection(Mongo initialMember) {
        // now loop and wait until the replica set is up.
        // its only considered up after an election has taken place
        // so that one member is PRIMARY and all others will be SECONDARY
        boolean shownWaiting = false;
        boolean isUp = false;
        do {
            DBObject status = initialMember.getDB("admin").command(new BasicDBObject("replSetGetStatus", 1));
            List<DBObject> members = (List<DBObject>) status.get("members");
            if (members != null) {
                if (!shownWaiting) {
                    StringBuilder simpleList = new StringBuilder();
                    for (DBObject member : members) {
                        simpleList.append(member.get("name")).append("  ");
                    }
                    logger.info("Waiting for election to take place between " + simpleList.toString());
                    shownWaiting = true;
                }
                isUp = true;
                for (DBObject member : members) {
                    String state = (String) member.get("stateStr");
                    if ((!"PRIMARY".equals(state)) && (!"SECONDARY".equals(state))) {
                        isUp = false; // every member must be either PRIMARY or
                                      // SECONDARY
                        try {
                            Thread.sleep(20);
                        } catch (InterruptedException e) {
                            return initialMember;
                        }
                        break;
                    }
                }
            } else {
                try {
                    Thread.sleep(10);
                } catch (InterruptedException e) {
                    return initialMember;
                }
            }
        } while (!isUp);
        return initialMember;
    }

    /**
     * One member of the replica set will be created for each folder passed in.
     * All instances will be passed the same set of arguments.
     * 
     * @param dbFolder
     * @param arguments
     * @throws UnknownHostException
     * @throws InterruptedException
     */
    public Mongo startup(String[] dbFolders, String arguments, Long cappedSize, Integer slaveDelay, String dbName, String collectionName, boolean deleteOldData) throws UnknownHostException, InterruptedException {
        return buildTestCollection(new Mongo(launchEmptyReplicaSet(dbFolders, "testSet", slaveDelay, arguments, deleteOldData), mOptions), cappedSize, dbName, collectionName);
    }

}
