package com.ociweb.mongoDB.most;

/*
 * Copyright (c) 2011, Object Computing, Inc.
 * All Rights reserved
 * See the file license.txt for licensing information.
 */
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;

import com.mongodb.BasicDBObject;
import com.mongodb.DB;
import com.mongodb.DBObject;
import com.mongodb.Mongo;
import com.mongodb.ServerAddress;

/**
 * 
 * factory for launching sharded cluster or replica set or single instance.
 * 
 * @author Nathan Tippy
 * 
 */
public class MongoDBShardedClusterServerFactory extends MongoDBReplicaSetServerFactory {

    public MongoDBShardedClusterServerFactory(String mongoDBPath, String bindAddr, int basePort) {
        super(mongoDBPath, bindAddr, basePort);
    }

    public Mongo startup(final String[] dbFolders, String[] configFolders, final String arguments, final Integer replicas, final Integer slaveDelay, Integer chunkSize, Integer shardMask, String shardKey, String dbName, String collectionName, final boolean deleteOldData) throws UnknownHostException,
            InterruptedException {

        int folderCount = dbFolders.length;
        int shardMemberCount = folderCount / replicas;

        // start up config members
        AtomicInteger configServersUp = new AtomicInteger();
        List<String> configAddrList = new ArrayList<String>();
        for (String cfgFolder : configFolders) {// caution, same arguments also
                                                // passed to configsvr
            configAddrList.add(launchEmptySingle("mongod", cfgFolder, "--configsvr " + arguments, configServersUp, deleteOldData));
        }

        // start up shard members
        ExecutorService pool = Executors.newFixedThreadPool(shardMemberCount);
        final AtomicInteger testSetNameId = new AtomicInteger();
        final List<String> shardMembers = new ArrayList<String>();
        int i = 0;
        while (i < folderCount) {
            final int index = i;

            // start up each of the shards in parallel or we may be waiting a
            // while
            pool.execute(new Runnable() {
                @Override
                public void run() {
                    // if replicas is one then we are sharding with single
                    // instances not replica sets
                    if (replicas == 1) {
                        AtomicInteger serversUp = new AtomicInteger();
                        shardMembers.add(launchEmptySingle("mongod", dbFolders[index], "--shardsvr " + arguments, serversUp, deleteOldData));
                        awaitFor(serversUp, 1);
                    } else {
                        String testSetName = "testSet" + testSetNameId.getAndIncrement();
                        List<ServerAddress> list = launchEmptyReplicaSet(Arrays.copyOfRange(dbFolders, index, index + replicas), testSetName, slaveDelay, "--shardsvr " + arguments, deleteOldData);
                        shardMembers.add(testSetName + "/" + list.toString().replace("[", "").replace("]", "").replace(" ", ""));
                    }

                }
            });

            i += replicas;
        }
        pool.shutdown();
        pool.awaitTermination(10, TimeUnit.HOURS);

        // we need to wait on the config servers if they are not up by now.
        awaitFor(configServersUp, configFolders.length);

        // start up mongos server
        StringBuilder args = new StringBuilder("--configdb ");
        for (String cfg : configAddrList) {
            args.append(cfg).append(',');
        }
        args.setLength(args.length() - 1);
        args.append(' ');

        if (chunkSize != null) {
            args.append("--chunkSize ");
            args.append(chunkSize).append(' ');
        }

        args.append(arguments.replaceAll("--journal", ""));// do not send
                                                           // journal to mongos

        // one mongos will be launched for each of the config servers
        int mongosCount = configFolders.length;
        AtomicInteger serversUp = new AtomicInteger();
        List<ServerAddress> mongosAddresses = new ArrayList<ServerAddress>();
        String host = null;
        while (mongosAddresses.size() < mongosCount) {
            // note: mongos not mongod
            host = launchEmptySingle("mongos", null, args.toString(), serversUp, false); // no
                                                                                         // folder,
                                                                                         // nothing
                                                                                         // to
                                                                                         // delete.
            mongosAddresses.add(new ServerAddress(host));
        }
        awaitFor(serversUp, mongosCount);
        // create mongo to be used for configuration then returned to caller
        Mongo primary;
        if (mongosCount == 1) {
            primary = new Mongo(host, mOptions);
        } else {
            primary = new Mongo(mongosAddresses, mOptions);
        }

        // if the old data was deleted it will be necessary to reconfigure the
        // sharded cluster
        if (deleteOldData) {
            DB adminDB = primary.getDB("admin");
            logger.info("mongo> use admin");
            // must configure shard before returning
            List<String> shardNames = new ArrayList<String>();
            for (String addr : shardMembers) {
                BasicDBObject cmd = new BasicDBObject("addshard", addr);
                logger.info("mongo> db.runCommand(" + cmd + ")");
                shardNames.add((String) logResult(primary, adminDB.command(cmd), cmd).get("shardAdded"));
            }

            // capped collections are not supported in shards
            buildTestCollection(primary, null, dbName, collectionName);
            String collectionNameSpace = dbName + "." + collectionName;

            // sharding setup
            BasicDBObject cmd = new BasicDBObject("enablesharding", dbName);
            logger.info("mongo> db.runCommand(" + cmd + ")");
            logResult(primary, adminDB.command(cmd), cmd);

            cmd = new BasicDBObject("shardcollection", collectionNameSpace).append("key", new BasicDBObject(shardKey, 1));
            logger.info("mongo> db.runCommand(" + cmd + ")");
            logResult(primary, adminDB.command(cmd), cmd);

            // if shardMod is null then auto sharding will be used without any
            // presplitting.
            if (shardMask != null) {
                // NOTE: shardMod should be much larger than initial
                // shardMemberCount to ensure that auto balancing can take place
                // later
                double shardRangeSize = (shardMask + 1) / (double) shardMemberCount;

                // must tell the collection about each of the split points which
                // define
                // the lines between each of the shards
                int s = shardMemberCount;
                while (--s > 0) {// there are 1 fewer split points than there
                                 // are shards
                    DBObject command = (new BasicDBObject("split", collectionNameSpace).append("middle", new BasicDBObject(shardKey, shardRangeSize * s)));
                    logger.info("mongo> db.runCommand(" + command + ")");
                    logResult(primary, adminDB.command(command), command);
                }

                // these new shards all exits in shard0000 so we need to move
                // all but
                // the first one to other shard instances
                s = shardMemberCount;

                while (--s >= 0) {// there are 1 fewer split points than there
                                  // are shards
                    // defining any object in the find which would if existed
                    // belong to
                    // that group is all that is needed to grab the entire group
                    // and move it to the 'to' destination shard
                    DBObject command = (new BasicDBObject("moveChunk", dbName + "." + collectionName).append("find", new BasicDBObject(shardKey, shardRangeSize * (s + .5d))).append("to", shardNames.get(s)));
                    logger.info("mongo> db.runCommand(" + command + ")");
                    // this error happens because we are not bothering to check
                    // first, there is no harm in ignoring it
                    String ignoreError = "that chunk is already on that shard";
                    logResult(primary, adminDB.command(command), command, ignoreError);
                }
            }
        }
        return primary;
    }

    public int mongosRequiredConnectionsCount(int shards, int replicaSetNodes, int clientConnections) {
        if (shards == replicaSetNodes) {
            // not using replica sets behind shards
            return (1 + shards * clientConnections);
        } else {
            return (1 + (shards * replicaSetNodes) * clientConnections);// from
                                                                        // MongoDB
                                                                        // sharding
                                                                        // FAQ
        }
    }

}
