package com.ociweb.mongoDB.most;

/*
 * Copyright (c) 2011, Object Computing, Inc.
 * All Rights reserved
 * See the file license.txt for licensing information.
 */
import java.util.concurrent.atomic.AtomicInteger;

import com.mongodb.Mongo;

/**
 * 
 * Simple factory class for launching a single mongodb server instance.
 * 
 * @author Nathan Tippy
 * 
 */
public class MongoDBSingleInstanceServerFactory extends MongoDBBaseServerFactory {

    public MongoDBSingleInstanceServerFactory(String mongoDBPath, String bindAddr, int basePort) {
        super(mongoDBPath, bindAddr, basePort);
    }

    /**
     * 
     * @param dbFolder
     * @param arguments
     * @return host address
     */
    protected String launchEmptySingle(String executable, String dbFolder, String arguments, AtomicInteger serversUp, boolean deleteOldData) {
        // ensure the folder exists and its empty
        validateTargetFolderForMongoDB(dbFolder, deleteOldData);

        int myPort;

        // sharded cluster will startup each shard in parallel so we need to
        // protect this
        synchronized (this) {
            myPort = (nextPort++);
        }

        // start up single server instance
        StringBuilder cmd = new StringBuilder();
        cmd.append(mongoDBPath).append(separator).append("bin").append(separator);
        cmd.append(executable);
        if (dbFolder != null) {// use default if path is not provided
            cmd.append(" --dbpath ").append(dbFolder);
        }
        ;

        if (bindAddr != null) {
            cmd.append(" --bind_ip ").append(bindAddr);
            if (!bindAddr.contains("127.0.0.1")) {
                cmd.append(",127.0.0.1");// required for shutdown request later
            }
        }

        cmd.append(" --port ").append(myPort);
        cmd.append(' ').append(arguments);

        launchMongoDBInstance(myPort, cmd.toString(), serversUp);
        return bindAddr + ":" + myPort;
    }

    public Mongo startup(String dbFolder, String arguments, Long cappedSize, String dbName, String collectionName, boolean deleteOldData) {

        AtomicInteger serversUp = new AtomicInteger();
        String host = launchEmptySingle("mongod", dbFolder, arguments, serversUp, deleteOldData);

        // wait for this single server to start accepting connections
        awaitFor(serversUp, 1);

        try {
            if (deleteOldData) {
                // create new Mongo object pointing to this new single server
                // instance
                return buildTestCollection(new Mongo(host, mOptions), cappedSize, dbName, collectionName);
            } else {
                return new Mongo(host, mOptions);
            }

        } catch (Exception e) {
            logger.throwing(getClass().getName(), "startup", e);
            shutdownServers();
            System.exit(-1);
            return null;
        }
    }
}
