// http://sites.google.com/site/mandelbox/what-is-a-mandelbox
// http://www.fractalforums.com/3d-fractal-generation/a-mandelbox-distance-estimate-formula/


typedef struct Matrix33 {
	float3 c0, c1, c2;
} Matrix33;



// matrix in column-major order to facilitate vector operations
Matrix33 MultiplyMM(Matrix33 A, Matrix33 B) {
	Matrix33 C;
	C.c0 = A.c0 * B.c0.x + A.c1 * B.c0.y + A.c2 * B.c0.z;
	C.c1 = A.c0 * B.c1.x + A.c1 * B.c1.y + A.c2 * B.c1.z;
	C.c2 = A.c0 * B.c2.x + A.c1 * B.c2.y + A.c2 * B.c2.z;
	return C;
}




float3 MultiplyMV(Matrix33 A, float3 B) {
	float3 t0 = A.c0 * B.x; 
	float3 t1 = A.c1 * B.y; 
	float3 t2 = A.c2 * B.z; 
	return (float3)(t0.x+t1.x+t2.x, t0.y+t1.y+t2.y, t0.z+t1.z+t2.z);
}



Matrix33 RotateX(float a) {
	float c = cos(a);
	float s = sin(a);
	Matrix33 m;
	m.c0 = (float3)(1,0,0);
	m.c1 = (float3)(0,c,s);
	m.c2 = (float3)(0,-s,c);
	return m;
}

Matrix33 RotateY(float a) {
	float c = cos(a);
	float s = sin(a);
	Matrix33 m;
	m.c0 = (float3)(c,0,-s);
	m.c1 = (float3)(0,1,0);
	m.c2 = (float3)(s,0,c);
	return m;
}

Matrix33 RotateZ(float a) {
	float c = cos(a);
	float s = sin(a);
	Matrix33 m;
	m.c0 = (float3)(c,s,0);
	m.c1 = (float3)(-s,c,0);
	m.c2 = (float3)(0,0,1);
	return m;
}








// BE SURE THIS IS AN EXACT MATCH TO C++
// http://forums.amd.com/forum/messageview.cfm?catid=390&threadid=128520  (without packed attribute, need padding in arbitrary places in the struct)
typedef struct __attribute__((packed)) Params {
	// rendering parameters
	float3 _cameraPosition;
	float3 _cameraOrientation;
	float3 _objectOrientation;
	int _antialiasing;
	int _width;  // overall image dimensions
	int _height;
	float _cameraZoom;
	float3 _ambient;
	float3 _lightPos;
	float3 _lightColor;
	float _maxRayLength;  
	float _shadows;
	int _type;
	int _fog;
	float3 _fogColor;
	float _fogMultiplier;
	float _fogPower;

	// Mandelbox parameters
	float _scale;  // mandelbox scale factor
	int _maxIterations;
	int _maxSteps;
	float3 _boxFoldColor;
	float3 _sphereFoldColor;
	float3 _specularColor;
	float _specularExponent;

	// additional parameters for the Mandelbulb 
	float _bailout;
	int _useJulia;
	float3 _julia_c;
	float _power;

	// set by Initialize
	float _aspectRatio;
	Matrix33 _viewRotation;
	Matrix33 _objRotation;
	float3 _eye;
	float _sampleStep;
	float _sampleContribution;
} Params;


int sgn(float f) {
	return (f<0.0f)?-1:1;
}



void Mandelbulb_DistanceEstimation(__global struct Params* params, const float3 *z0, float *distance, float3 *color) {
	*distance = 0.0f;
	if (color!=0)
		*color = (float3)(1.0f,1.0f,1.0f);

	float2 phase = (float2)(0.0f,0.0f);

	float power = params->_power;
	float3 c = (params->_useJulia==1) ? params->_julia_c : *z0; // Julia set has fixed c, Mandelbrot c changes with location
	float3 z = *z0;
	float pd = power - 1.0f;			 // power for derivative

	// Convert z to polar coordinates
	float r	 = length(z);
	float th = atan2(z.y, z.x);
	float ph = 0.0f;
	if (r != 0.0f)
		ph = asin(z.z / r);

	float3 dz;
	float ph_dz = 0.0f;
	float th_dz = 0.0f;
	float r_dz	= 1.0f;
	float powR, powRsin;

	int maxIterations = params->_maxIterations;
	float bailout = params->_bailout;

	// Iterate to compute the distance estimator.
	for (int n = 0; n < maxIterations; n++) {
		// Calculate derivative of
		powR = power * pow(r, pd);
		powRsin = powR * r_dz * sin(ph_dz + pd*ph);
		dz.x = powRsin * cos(th_dz + pd*th) + 1.0f;
		dz.y = powRsin * sin(th_dz + pd*th);
		dz.z = powR * r_dz * cos(ph_dz + pd*ph);

		// polar coordinates of derivative dz
		r_dz  = length(dz);
		th_dz = atan2(dz.y, dz.x);
		ph_dz = acos(dz.z / r_dz);

		// z iteration
		powR = pow(r, power);
		powRsin = sin(power*ph);
		z.x = powR * powRsin * cos(power*th);
		z.y = powR * powRsin * sin(power*th);
		z.z = powR * cos(power*ph);
		z += c;

		r  = length(z);
		if (r > bailout) break;

		th = atan2(z.y, z.x) + phase.x;
		ph = phase.y;
		if (r!=0.0f)
			ph = acos(z.z / r) + phase.y;
	}

	// Return the distance estimation value which determines the next raytracing
	// step size, or if whether we are within the threshold of the surface.
	if (r==0.0f)
		*distance=0.0f;
	else
		*distance= 0.5f * r * log(r)/r_dz;

	if (color != 0) {
		/*
		(*color).x = (z.x>0)?1.0:0.25;
		(*color).y = (z.y>0)?1.0:0.25;
		(*color).z = (z.z>0)?1.0:0.25;
		*/
		/*
		(*color).x = (sgn(z.x)!=sgn((*z0).x))?1.0:0.25;
		(*color).y = (sgn(z.y)!=sgn((*z0).y))?1.0:0.25;
		(*color).z = (sgn(z.z)!=sgn((*z0).z))?1.0:0.25;
		*/
	}
}

//can look inside?


__constant float fixedRadius2 = 1.0f * 1.0f;
__constant float minRadius2 = 0.5f * 0.5f;

void Mandelbox_DistanceEstimation(__global struct Params* params, const float3 *z0, float *distance, float3 *color) {
	*distance = 0.0f;
	if (color!=0)
		*color = 0;

	float3 c = *z0;
	float3 z = *z0;
	float factor = params->_scale;
	float scale = params->_scale;
	float3 boxFoldColor = params->_boxFoldColor;
	float3 sphereFoldColor = params->_sphereFoldColor;
	int maxIterations = params->_maxIterations;

	int boxFoldCount = 0;
	int sphereFoldCount = 0;
	int n;
	for (n = 0; n < maxIterations; n++) {
		if (z.x > 1.0f) { z.x = 2.0f - z.x; boxFoldCount++; }
		else if (z.x < -1.0f) { z.x = -2.0f - z.x; boxFoldCount++; }

		if (z.y > 1.0f) { z.y = 2.0f - z.y; boxFoldCount++; }
		else if (z.y < -1.0f) { z.y = -2.0f - z.y; boxFoldCount++; }

		if (z.z > 1.0f) { z.z = 2.0f - z.z; boxFoldCount++; }
		else if (z.z < -1.0f) { z.z = -2.0f - z.z; boxFoldCount++; }

		float r = length(z);
		float r2 = r*r;

		if (r2 < minRadius2) {
			z = (z * fixedRadius2) / minRadius2;
			factor = (factor * fixedRadius2) / minRadius2;
			sphereFoldCount++;
		}
		else if (r2 < fixedRadius2) {
			z = (z * fixedRadius2) / r2;
			factor = (factor * fixedRadius2) / r2;
			sphereFoldCount++;
		}

		z = (z * scale) + c;
		factor *= scale;
		r = length(z);

		*distance = r / fabs(factor);
		if (r > 1024)
			break;
	}

	if (color != 0) {
		float totalFolds = boxFoldCount + sphereFoldCount;
		*color = (boxFoldCount/totalFolds)*boxFoldColor + (sphereFoldCount/totalFolds)*sphereFoldColor;
		*color = clamp(*color, 0, 1);

		/*
		// youtube animation
		if (z.x<0) (*color).x=1.0f-(*color).x;
		if (z.y<0) (*color).y=1.0f-(*color).y;
		if (z.z<0) (*color).z=1.0f-(*color).z;
		*/
	}

	if (n == maxIterations)
		*distance=0;
}



void DistanceEstimation(__global struct Params* params, const float3 *z0, float *distance, float3 *color) {
	if (params->_type == 1)
	    Mandelbox_DistanceEstimation(params, z0, distance, color);
	else
		Mandelbulb_DistanceEstimation(params, z0, distance, color);
}




float3 Normal(__global struct Params* params, float3 z, float e) {
	float3 xoff = (float3)(e, 0.0f, 0.0f), yoff = (float3)(0.0f, e, 0.0f), zoff = (float3)(0.0f, 0.0f, e);

	float3 zxp = z+xoff, zxm = z-xoff;
	float3 zyp = z+yoff, zym = z-yoff;
	float3 zzp = z+zoff, zzm = z-zoff;
	float dxp, dxm, dyp, dym, dzp, dzm;
	DistanceEstimation(params, &zxp, &dxp, 0);
	DistanceEstimation(params, &zxm, &dxm, 0);
	DistanceEstimation(params, &zyp, &dyp, 0);
	DistanceEstimation(params, &zym, &dym, 0);
	DistanceEstimation(params, &zzp, &dzp, 0);
	DistanceEstimation(params, &zzm, &dzm, 0);

	float3 d = (float3)(dxp-dxm, dyp-dym, dzp-dzm);
	return normalize(d / (2*e));
}



bool InShadow(__global struct Params* params, float3 ray, float3 lightDirection, float eps) {
	float distance = 4.0f;
	float maxRayLength2 = params->_maxRayLength*params->_maxRayLength;
	for (int j = 0; j < params->_maxSteps; ++j) {
		DistanceEstimation(params, &ray, &distance, 0);

		// March ray forward
		ray += distance * lightDirection;

		// Are we within the intersection threshold or completely missed the fractal
		if (distance < eps || dot(ray, ray) > maxRayLength2) break;
	}
	return distance < eps;
}


float3 interp(float3 a, float3 b, float t) {
	return (1.0-t)*a + t*b;
}


float3 RenderPixel(__global struct Params* params, float3 rayDirection) {
	float rayLength = 0;
	const float EPSILON = 1.0E-6f;
	float eps = EPSILON;
	float distance;
	float3 diffuse;
	bool intersected = false;
	float3 ray = params->_eye;
	float maxRayLength = params->_maxRayLength;

	for (int i = 0; i < params->_maxSteps; ++i) {
		DistanceEstimation(params, &ray, &distance, &diffuse);
		ray += rayDirection*distance;
		rayLength += distance;

		if (distance < eps) {
			intersected = true;  // hit the fractal
			break;
		}
		if (rayLength > maxRayLength)
			break;  // exceeded max ray length	

		// pixel scale of 1/1024
		eps = max(EPSILON, (1.0f / 1024)*rayLength);
	}
   

    float3 pixelColor = 0;  // background is black, and alpha 0 (transparent)
	if (intersected) {
		// normal at the intersection point
		float3 N = Normal(params, ray, eps/2);
		
		// compute color (single white light source of intensity 1)
		float3 L = normalize(MultiplyMV(params->_objRotation, params->_lightPos) - ray);
		float NdotL = dot(N, L);
		float3 color = 0.0f;
		float3 specular = 0.0f;
		if (NdotL > 0.0f) {
			color = diffuse * params->_lightColor * NdotL; 

			// Phong highlight
			float3 E = normalize(params->_eye - ray);		// find the vector to the eye
			float3 R = L - 2.0f * NdotL * N;		// find the reflected vector
			float RdE = dot(R,E);

			if (RdE <= 0.0f)
				specular = params->_specularColor * pow(fabs(RdE), params->_specularExponent);

			if (params->_shadows > 0.0f) {
				// The shadow ray will start at the intersection point and go
				// towards the point light. We initially move the ray origin
				// a little bit along this direction so that we don't mistakenly
				// find an intersection with the same point again.
				float3 lightDirection = normalize(MultiplyMV(params->_objRotation, params->_lightPos - ray));
				ray += N * eps * 2.0f;

				if (InShadow(params, ray, lightDirection, eps)) {
					color *= (1.0f - params->_shadows);
				}
				else
					color += specular;
			}
			else
				color += specular;
		}

		pixelColor = clamp(color + params->_ambient, 0, 1);
	}

	if (params->_fog)
		pixelColor.xyz=interp(pixelColor.xyz, params->_fogColor, clamp(pow((rayLength/maxRayLength)*params->_fogMultiplier, params->_fogPower), 0, 1));

	return pixelColor;
}




float3 RayDirection(__global struct Params* params, float2 pixel) {
	float3 rayDirection = (float3)(
		2.0f * params->_aspectRatio * pixel.x / (float)params->_width - params->_aspectRatio,
		-2.0f * pixel.y / (float)params->_height + 1.0f,
		-2.0f * exp(params->_cameraZoom));
	return normalize(MultiplyMV(params->_objRotation, MultiplyMV(params->_viewRotation, rayDirection)));
}


__kernel void Initialize(__global struct Params* params) {
	params->_aspectRatio = ((float)params->_width) / ((float)params->_height);

	// Camera orientation
	Matrix33 viewRotationX, viewRotationY, viewRotationZ;
	viewRotationY = RotateY(radians(-params->_cameraOrientation.x));
	viewRotationZ = RotateZ(radians(-params->_cameraOrientation.y));
	viewRotationX = RotateX(radians(-params->_cameraOrientation.z));
	params->_viewRotation=MultiplyMM(MultiplyMM(viewRotationX, viewRotationY), viewRotationZ);

	// Object orientation
	Matrix33 objRotationX, objRotationY, objRotationZ;
	objRotationY = RotateY(radians(-params->_objectOrientation.x));
	objRotationZ = RotateZ(radians(-params->_objectOrientation.y));
	objRotationX = RotateX(radians(-params->_objectOrientation.z));
	params->_objRotation=MultiplyMM(MultiplyMM(objRotationX, objRotationY), objRotationZ);

	params->_eye = params->_cameraPosition;
	// if (eye == (float3)(0, 0, 0)) eye = (float3)(0, 0.0001, 0);

	params->_eye = MultiplyMV(params->_objRotation, params->_eye);

	params->_sampleStep = 1.0f / (float)(params->_antialiasing);
	params->_sampleContribution = 1.0f / pow((float)(params->_antialiasing), 2.0f);
}



__kernel void Mandelbox(__global struct Params* params, __global uchar4 *imageBuffer, int py) {
	float3 color = 0;

	int px = get_global_id(0);

	if (params->_antialiasing > 1) {
		for (float i = 0.0f; i < 1.0f; i += params->_sampleStep)
			for (float j = 0.0f; j < 1.0f; j += params->_sampleStep) {
				color += params->_sampleContribution * RenderPixel(params, RayDirection(params, (float2)(px + i, py + j)));
			}
	} else
		color = RenderPixel(params, RayDirection(params, (float2)(px, py)));

	color = clamp(color, 0, 1);

	imageBuffer[px] = (uchar4)((uchar)(color.x*255), (uchar)(color.y*255), (uchar)(color.z*255), 255);
}

/*
C:\Users\calkinsc\AppData\Local\Temp\OCLB63C.tmp.cl(281): warning: argument of
         type "constant Matrix33 *" is incompatible with parameter of type
         "Matrix33 *"

C:\Users\calkinsc\AppData\Local\Temp\OCL24A6.tmp.cl(319): error: argument of
          type "global Matrix33 *" is incompatible with parameter of type
          "Matrix33 *"
*/

/*

[ need to incorporate blue marble converter as well to project ]   


*/

