#include "TestRunner.h"
#include "ImageBuilder.h"
#include "FileManagement.h"
#include "Crop.h"
#include "Half.h"
#include "RowCache.h"
#include "Tile.h"
#include <fstream>

template <typename T>
void CreateImage(const std::string &fileName, T value, unsigned int width, unsigned int height, 
	unsigned int channelCount, unsigned int channelDepth) {
	ImageWriterPtr writer = ImageBuilder::GetImageBuilder(fileName)->GetWriter();
	writer->Open(fileName, width, height, channelCount, channelDepth);
	ImageRow row = writer->GetImageRow();
	for (unsigned int y=0; y<height; y++) {
		for (unsigned int x=0; x<width; x++) 
			row.SetPixel(x, value, value, value, value);
		writer->WriteRow(row);
	}
	writer->Close();
}

void VerifyImageInfo(const std::string &fileName, 
	unsigned int width, unsigned int height, unsigned int channelCount, unsigned int channelDepth) {
	ImageReaderPtr reader = ImageBuilder::GetImageBuilder(fileName)->GetReader();
	reader->Open(fileName);
	ASSERT_EQUAL(width, reader->GetWidth());
	ASSERT_EQUAL(height, reader->GetHeight());
	ASSERT_EQUAL(channelCount, reader->GetChannelCount());
	ASSERT_EQUAL(channelDepth, reader->GetChannelDepth());
}

template <typename T>
void VerifyPixelValue(const std::string &fileName, T value, unsigned int channelCount) {
	ImageReaderPtr reader = ImageBuilder::GetImageBuilder(fileName)->GetReader();
	reader->Open(fileName);
	ImageRow row = reader->GetImageRow();
	for (unsigned int y=0; y<reader->GetHeight(); y++) {
		reader->ReadNextRow(row);
		T v1, v2, v3, v4;
		for (unsigned int x=0; x<reader->GetWidth(); x++) {
			row.GetPixel(x, v1, v2, v3, v4);
			ASSERT_EQUAL(value, v1);
			ASSERT_EQUAL(value, v2);
			ASSERT_EQUAL(value, v3);
			if (channelCount == 4)
				ASSERT_EQUAL(value, v4);
		}
	}
	reader->Close();
}

void VerifyFileExists(const std::string &fileName) {
  std::ifstream src(fileName.c_str(), std::ios::binary | std::ios::in);
	ASSERT_TRUE(src.is_open());
}


class RowCacheTest : public RowCache {
public:
	RowCacheTest(std::string fileName, unsigned int maxCacheSize) : RowCache(fileName, maxCacheSize) {}
	unsigned int GetLastRowRead() const { return _lastRowRead; }
	const std::map<unsigned int, ImageRow> &GetRowCache() const { return _rowCache; }
	const std::list<unsigned int> &GetMRU() const { return _mru; }
};


void testCreatePNG_3_8() {
	CreateImage<unsigned char>("test.png", 255, 512, 256, 3, 8);
	VerifyImageInfo("test.png", 512, 256, 3, 8);
}

void testCreatePNG_3_16() {
	CreateImage<unsigned char>("test.png", 255, 512, 256, 3, 16);
	VerifyImageInfo("test.png", 512, 256, 3, 16);
}

void testCreatePNG_4_8() {
	CreateImage<unsigned char>("test.png", 255, 512, 256, 4, 8);
	VerifyImageInfo("test.png", 512, 256, 4, 8);
}

void testCreatePNG_4_16() {
	CreateImage<unsigned char>("test.png", 255, 512, 256, 4, 16);
	VerifyImageInfo("test.png", 512, 256, 4, 16);
}

void testPNGChar() {
	CreateImage<unsigned char>("test.png", 123, 512, 256, 4, 16);
	VerifyPixelValue<unsigned char>("test.png", 123, 4);
}

void testPNGShort() {
	CreateImage<unsigned short>("test.png", 12345, 512, 256, 4, 16);
	VerifyPixelValue<unsigned short>("test.png", 12345, 4);
}

void testCreateJPG_3_8() {
	CreateImage<unsigned char>("test.jpg", 255, 512, 256, 3, 8);
	VerifyImageInfo("test.jpg", 512, 256, 3, 8);
}

void testCrop_3_8() {
	CreateImage<unsigned char>("test.png", 255, 512, 256, 3, 8);
	RowCache rowCache("test.png", 20);
	Crop(rowCache, 5,5, 100,100, "crop.png");
	VerifyImageInfo("crop.png", 100, 100, 3, 8);
	VerifyPixelValue<unsigned char>("crop.png", 255, 3);
}

void testCrop_3_16() {
	CreateImage<unsigned char>("test.png", 255, 512, 256, 3, 16);
	RowCache rowCache("test.png", 20);
	Crop(rowCache, 5,5, 100,100, "crop.png");
	VerifyImageInfo("crop.png", 100, 100, 3, 16);
	VerifyPixelValue<unsigned char>("crop.png", 255, 3);
}

void testCrop_4_8() {
	CreateImage<unsigned char>("test.png", 255, 512, 256, 4, 8);
	RowCache rowCache("test.png", 20);
	Crop(rowCache, 5,5, 100,100, "crop.png");
	VerifyImageInfo("crop.png", 100, 100, 4, 8);
	VerifyPixelValue<unsigned char>("crop.png", 255, 4);
}

void testCrop_4_16() {
	CreateImage<unsigned char>("test.png", 255, 512, 256, 4, 16);
	RowCache rowCache("test.png", 20);
	Crop(rowCache, 5,5, 100,100, "crop.png");
	VerifyImageInfo("crop.png", 100, 100, 4, 16);
	VerifyPixelValue<unsigned char>("crop.png", 255, 4);
}

void testHalfEven_3_8() {
	CreateImage<unsigned char>("test.png", 255, 512, 256, 3, 8);
	Half("test.png", "half.png", 0);
	VerifyImageInfo("half.png", 256, 128, 3, 8);
	VerifyPixelValue<unsigned char>("half.png", 255, 3);
}

void testHalfOdd_3_8() {
	CreateImage<unsigned char>("test.png", 255, 513, 257, 3, 8);
	Half("test.png", "half.png", 0);
	VerifyImageInfo("half.png", 257, 129, 3, 8);  // size is rounded up if odd
	VerifyPixelValue<unsigned char>("half.png", 255, 3);
}

void testHalfEven_4_8() {
	CreateImage<unsigned char>("test.png", 255, 512, 256, 4, 8);
	Half("test.png", "half.png", 0);
	VerifyImageInfo("half.png", 256, 128, 4, 8);
	VerifyPixelValue<unsigned char>("half.png", 255, 4);
}

void testHalfOdd_4_8() {
	CreateImage<unsigned char>("test.png", 255, 513, 257, 4, 8);
	Half("test.png", "half.png", 0);
	VerifyImageInfo("half.png", 257, 129, 4, 8);  // size is rounded up if odd
	VerifyPixelValue<unsigned char>("half.png", 255, 4);
}

void testHalfEven_3_16() {
	CreateImage<unsigned char>("test.png", 255, 512, 256, 3, 16);
	Half("test.png", "half.png", 0);
	VerifyImageInfo("half.png", 256, 128, 3, 16);
	VerifyPixelValue<unsigned char>("half.png", 255, 3);
}

void testHalfOdd_3_16() {
	CreateImage<unsigned char>("test.png", 255, 513, 257, 3, 16);
	Half("test.png", "half.png", 0);
	VerifyImageInfo("half.png", 257, 129, 3, 16);  // size is rounded up if odd
	VerifyPixelValue<unsigned char>("half.png", 255, 3);
}

void testHalfEven_4_16() {
	CreateImage<unsigned char>("test.png", 255, 512, 256, 4, 16);
	Half("test.png", "half.png", 0);
	VerifyImageInfo("half.png", 256, 128, 4, 16);
	VerifyPixelValue<unsigned char>("half.png", 255, 4);
}

void testHalfOdd_4_16() {
	CreateImage<unsigned char>("test.png", 255, 513, 257, 4, 16);
	Half("test.png", "half.png", 0);
	VerifyImageInfo("half.png", 257, 129, 4, 16);  // size is rounded up if odd
	VerifyPixelValue<unsigned char>("half.png", 255, 4);
}


void testRowCache() {
	std::list<unsigned int>::const_iterator i;
	CreateImage<unsigned char>("test.png", 255, 512, 256, 3, 8);
	RowCacheTest cache("test.png", 2);

	// initially no rows read, and cache is empty
	ASSERT_EQUAL(-1, cache.GetLastRowRead());
	ASSERT_EQUAL(0, cache.GetRowCache().size());
	ASSERT_EQUAL(0, cache.GetMRU().size());

	// get row 0
	ImageRow r = cache.GetRow(0); 

	// row 0 was the last row read, row 0 is the only row cached
	ASSERT_EQUAL(0, cache.GetLastRowRead());
	ASSERT_EQUAL(1, cache.GetRowCache().size());
	ASSERT_EQUAL(1, cache.GetMRU().size());
	i = cache.GetMRU().cbegin();
	ASSERT_EQUAL(0, *i);  // 0 is the most recent row

	// get row 0 again
	r = cache.GetRow(0); 

	// row 0 was the last row read, row 0 is the only row cached
	ASSERT_EQUAL(0, cache.GetLastRowRead());

	ASSERT_EQUAL(1, cache.GetRowCache().size());
	ASSERT_EQUAL(1, cache.GetMRU().size());
	i = cache.GetMRU().cbegin();
	ASSERT_EQUAL(0, *i);  // 0 is the most recent row

	// get row 1
	r = cache.GetRow(1); 

	// row 1 was the last row read, rows 0 and 1 are cached
	ASSERT_EQUAL(1, cache.GetLastRowRead());

	ASSERT_EQUAL(2, cache.GetRowCache().size());
	ASSERT_EQUAL(2, cache.GetMRU().size());
	i = cache.GetMRU().cbegin();
	ASSERT_EQUAL(1, *i++);  // 1 is the most recent row
	ASSERT_EQUAL(0, *i);  // 0 is the 2nd most recent row

	// get row 0 again
	r = cache.GetRow(0); 

	// row 1 was the last row read (0 is still in the cache), rows 0 and 1 are cached
	ASSERT_EQUAL(1, cache.GetLastRowRead());

	ASSERT_EQUAL(2, cache.GetRowCache().size());
	ASSERT_EQUAL(2, cache.GetMRU().size());
	i = cache.GetMRU().cbegin();
	ASSERT_EQUAL(0, *i++);  // 0 is the most recent row
	ASSERT_EQUAL(1, *i);  // 1 is the 2nd most recent row


	// get row 2
	r = cache.GetRow(2); 

	// row 2 was the last row read, row 1 is cached as 0 was discarded
	ASSERT_EQUAL(2, cache.GetLastRowRead());

	ASSERT_EQUAL(2, cache.GetRowCache().size());
	ASSERT_EQUAL(2, cache.GetMRU().size());
	i = cache.GetMRU().cbegin();
	ASSERT_EQUAL(2, *i++);  // 2 is the most recent row
	ASSERT_EQUAL(0, *i);  // 0 is the 2nd most recent row

	// get row 1
	r = cache.GetRow(1); 

	// row 0 was the last row read, row 1 is cached as 0 was discarded
	ASSERT_EQUAL(1, cache.GetLastRowRead());

	ASSERT_EQUAL(2, cache.GetRowCache().size());
	ASSERT_EQUAL(2, cache.GetMRU().size());
	i = cache.GetMRU().cbegin();
	ASSERT_EQUAL(1, *i++);  // 1 is the most recent row
	ASSERT_EQUAL(2, *i);  // 2 is the 2nd most recent row
}


void testCopy() {
	{
		std::ofstream out("test.dat", std::ios::binary);
		for (int i=0; i<20000; i++)
			out << (char)(i%128);
	}
	FileManagement::Copy("test.dat", "test2.dat");
	std::ifstream in("test2.dat", std::ios::binary);
	char buf[20000];
	in.read(buf, 20000);
	ASSERT_EQUAL(20000, in.gcount());
	for (int i=0; i<20000; i++)
		ASSERT_EQUAL((char)(i%128), buf[i]);
}

void testBasename() {
	ASSERT_EQUAL("fred", FileManagement::GetBasename("fred"));
	ASSERT_EQUAL("fred", FileManagement::GetBasename("fred.txt"));
	ASSERT_EQUAL("fred", FileManagement::GetBasename(std::string("dir1") + FileManagement::PathSep() + "fred.txt"));
	ASSERT_EQUAL("fred", FileManagement::GetBasename(std::string("dir1") + FileManagement::PathSep() + "dir2" + FileManagement::PathSep() + "fred.txt"));
	ASSERT_EQUAL("test1", FileManagement::GetBasename("test1"));
	ASSERT_EQUAL("test1", FileManagement::GetBasename("test1.png"));
	ASSERT_EQUAL("test1", FileManagement::GetBasename(std::string("dir1") + FileManagement::PathSep() + "test1.png"));
	ASSERT_EQUAL("test1", FileManagement::GetBasename(std::string("dir1") + FileManagement::PathSep() + "dir2" + FileManagement::PathSep() + "test1.png"));
}


void VerifyTilesExist(const std::string &base, const std::string &ext) {
	VerifyFileExists(base + "_files" + FileManagement::PathSep() + "0" + FileManagement::PathSep() + "0_0" + ext);
	VerifyFileExists(base + "_files" + FileManagement::PathSep() + "1" + FileManagement::PathSep() + "0_0" + ext);
	VerifyFileExists(base + "_files" + FileManagement::PathSep() + "2" + FileManagement::PathSep() + "0_0" + ext);
	VerifyFileExists(base + "_files" + FileManagement::PathSep() + "3" + FileManagement::PathSep() + "0_0" + ext);
	VerifyFileExists(base + "_files" + FileManagement::PathSep() + "4" + FileManagement::PathSep() + "0_0" + ext);
	VerifyFileExists(base + "_files" + FileManagement::PathSep() + "5" + FileManagement::PathSep() + "0_0" + ext);
	VerifyFileExists(base + "_files" + FileManagement::PathSep() + "6" + FileManagement::PathSep() + "0_0" + ext);
	VerifyFileExists(base + "_files" + FileManagement::PathSep() + "7" + FileManagement::PathSep() + "0_0" + ext);
	VerifyFileExists(base + "_files" + FileManagement::PathSep() + "8" + FileManagement::PathSep() + "0_0" + ext);
	VerifyFileExists(base + "_files" + FileManagement::PathSep() + "9" + FileManagement::PathSep() + "0_0" + ext);
	VerifyFileExists(base + "_files" + FileManagement::PathSep() + "9" + FileManagement::PathSep() + "0_1" + ext);
	VerifyFileExists(base + "_files" + FileManagement::PathSep() + "9" + FileManagement::PathSep() + "1_0" + ext);
	VerifyFileExists(base + "_files" + FileManagement::PathSep() + "9" + FileManagement::PathSep() + "1_1" + ext);
	VerifyFileExists(base + "_files" + FileManagement::PathSep() + "10" + FileManagement::PathSep() + "0_0" + ext);
	VerifyFileExists(base + "_files" + FileManagement::PathSep() + "10" + FileManagement::PathSep() + "0_1" + ext);
	VerifyFileExists(base + "_files" + FileManagement::PathSep() + "10" + FileManagement::PathSep() + "0_2" + ext);
	VerifyFileExists(base + "_files" + FileManagement::PathSep() + "10" + FileManagement::PathSep() + "0_3" + ext);
	VerifyFileExists(base + "_files" + FileManagement::PathSep() + "10" + FileManagement::PathSep() + "1_0" + ext);
	VerifyFileExists(base + "_files" + FileManagement::PathSep() + "10" + FileManagement::PathSep() + "1_1" + ext);
	VerifyFileExists(base + "_files" + FileManagement::PathSep() + "10" + FileManagement::PathSep() + "1_2" + ext);
	VerifyFileExists(base + "_files" + FileManagement::PathSep() + "10" + FileManagement::PathSep() + "1_3" + ext);
	VerifyFileExists(base + "_files" + FileManagement::PathSep() + "10" + FileManagement::PathSep() + "2_0" + ext);
	VerifyFileExists(base + "_files" + FileManagement::PathSep() + "10" + FileManagement::PathSep() + "2_1" + ext);
	VerifyFileExists(base + "_files" + FileManagement::PathSep() + "10" + FileManagement::PathSep() + "2_2" + ext);
	VerifyFileExists(base + "_files" + FileManagement::PathSep() + "10" + FileManagement::PathSep() + "2_3" + ext);
	VerifyFileExists(base + "_files" + FileManagement::PathSep() + "10" + FileManagement::PathSep() + "3_0" + ext);
	VerifyFileExists(base + "_files" + FileManagement::PathSep() + "10" + FileManagement::PathSep() + "3_1" + ext);
	VerifyFileExists(base + "_files" + FileManagement::PathSep() + "10" + FileManagement::PathSep() + "3_2" + ext);
	VerifyFileExists(base + "_files" + FileManagement::PathSep() + "10" + FileManagement::PathSep() + "3_3" + ext);
}

void testTile() {
	CreateImage<unsigned char>("test1.png", 255, 1000, 1000, 3, 8);
	Tile("test1.png", "", 256, 1, 0);
	VerifyTilesExist("test1", ".png");
	
	CreateImage<unsigned char>("test2.jpg", 255, 1000, 1000, 3, 8);
	Tile("test2.jpg", "", 256, 1, 0);
	VerifyTilesExist("test2", ".jpg");

	CreateImage<unsigned char>("test3.png", 255, 1000, 1000, 3, 8);
	Tile("test3.png", ".jpg", 256, 1, 0);
	VerifyTilesExist("test3", ".jpg");
	
	CreateImage<unsigned char>("test4.jpg", 255, 1000, 1000, 3, 8);
	Tile("test4.jpg", ".png", 256, 1, 0);
	VerifyTilesExist("test4", ".png");
}


int main() {
	TestRunner r;
	ADD_TEST(r, testCreatePNG_3_8);
	ADD_TEST(r, testCreatePNG_3_16);
	ADD_TEST(r, testCreatePNG_4_8);
	ADD_TEST(r, testCreatePNG_4_16);
	ADD_TEST(r, testPNGChar);
	ADD_TEST(r, testPNGShort);
	ADD_TEST(r, testCreateJPG_3_8);
	ADD_TEST(r, testCrop_3_8);
	ADD_TEST(r, testCrop_3_16);
	ADD_TEST(r, testCrop_4_8);
	ADD_TEST(r, testCrop_4_16);
	ADD_TEST(r, testHalfEven_3_8);
	ADD_TEST(r, testHalfOdd_3_8);
	ADD_TEST(r, testHalfEven_4_8);
	ADD_TEST(r, testHalfOdd_4_8);
	ADD_TEST(r, testHalfEven_3_16);
	ADD_TEST(r, testHalfOdd_3_16);
	ADD_TEST(r, testHalfEven_4_16);
	ADD_TEST(r, testHalfOdd_4_16);
	ADD_TEST(r, testRowCache);
	ADD_TEST(r, testCopy);
	ADD_TEST(r, testBasename);
	ADD_TEST(r, testTile);
	r.Run();
}

