#include "RowCache.h"
#include "ImageBuilder.h"
#include <algorithm>

RowCache::RowCache(const std::string &fileName, unsigned int maxCacheSize) : 
_fileName(fileName), _maxCacheSize(maxCacheSize), _lastRowRead(-1) {
	_reader = ImageBuilder::GetImageBuilder(fileName)->GetReader();
	_reader->Open(fileName);
}


ImageRow RowCache::GetRow(unsigned int row) {
	// if already cached, return it
  std::map<unsigned int, ImageRow>::iterator rowIter = _rowCache.find(row);
	if (rowIter != _rowCache.end()) {
		// make the requested row the most recently used row (move it to the head of the _mru)
		_mru.erase(std::remove(_mru.begin(), _mru.end(), row), _mru.end());  // remove it, whereever it is in the mru list
		_mru.push_front(row);  // add it at the front

		return rowIter->second;
	}

	// not found in the cache, so it needs to be read

	// first, check to see if there is room in the cache for the new row
	// if not, purge the oldest row (the one at the back)
	if (_mru.size() >=  _maxCacheSize) {
		unsigned int rowToPurge = _mru.back();
		// std::cout << "Purge " << rowToPurge << std::endl;
		_mru.pop_back();
		_rowCache.erase(rowToPurge);
	}

	// read the row from the file
	// if the row needed is past where the file has already been read, then restart the read from the beginning
	if ((_lastRowRead >= row) && (_lastRowRead != -1)) {
		_reader = ImageBuilder::GetImageBuilder(_fileName)->GetReader();
		_reader->Open(_fileName);
		_lastRowRead = -1;
	}

	// read and discard rows up to the desired row
	ImageRow rowPtr = _reader->GetImageRow();
	while ((_lastRowRead == -1) || (_lastRowRead < row)) {
		_reader->ReadNextRow(rowPtr);
		_lastRowRead++;
	}

	// the last row read is the desired row - cache it and return it
	// std::cout << "Read " << row << std::endl;

	// make the requested row the most recently used row (move it to the head of the _mru)
	_mru.erase(std::remove(_mru.begin(), _mru.end(), row), _mru.end());  // remove it, whereever it is in the mru list
	_mru.push_front(row);  // add it at the front

	// cache the row data itself
	_rowCache[row] = rowPtr;

	// return the row
	return rowPtr;
}
