#include "ImageBuilder.h"
#include "CommandLine.h"
#include "FileManagement.h"
#include <iostream>
#include <fstream>
#include <math.h>

// GRIDFLOAT data converter
// Description: see readme.txt in each image directory
// Data: http://seamless.usgs.gov/, specifically http://seamless.usgs.gov/ned13.php
// for Hawaii: http://gisdata.usgs.gov/XMLWebServices2/%28S%28i2z0guvjdywiit55ngaxko55%29%29/getTDDSDownloadURLs.aspx?XMin=-160.4&YMin=18.1&XMax=-154.6&YMax=23.0&EPSG=4326&STATE=Hawaii&COUNTY=#
// n20w156 is a good one

class FileReader {
	const unsigned int _rowWidth;
	std::vector<unsigned char> _rawRow;  // to save memory allocations

	// see http://stackoverflow.com/questions/1001307/detecting-endianness-programmatically-in-a-c-program
	bool IsBigEndianArchitecture() {
		union {
	#ifdef WIN32
			__int32 i;
	#else
			uint32_t i;
	#endif
			char c[4];
		} bint = {0x01020304};

		return bint.c[0] == 1; 
	}

	bool _isBigEndianArchitecture;
	std::ifstream _in;
public:
	FileReader(const std::string &fileName) : _rowWidth(10812), _in(fileName.c_str(), std::ios::in|std::ios::binary) {
		_rawRow.resize(4*_rowWidth);
		_isBigEndianArchitecture = IsBigEndianArchitecture();
	}

	bool ReadRow(std::vector<float> &row) {
		if (!_in.is_open() || !_in.good())
			return false;
		row.resize(_rowWidth);

		_in.read((char *)&_rawRow[0], 4*_rowWidth);
		for (unsigned int x=0; x<_rowWidth; x++) {
			unsigned char *ptr = (unsigned char *)&row[x];
			for (unsigned int i=0; i<4; i++)
				*ptr++ = _rawRow[x*4+(_isBigEndianArchitecture?i:3-i)];
		}
		return true;
	}
};

int main(int argc, char *argv[]) {
	try {
		CommandLine cl(argc, argv);
		if (cl.Length() == 0) {
			std::cout << "Usage: gridfloat -d data.flt -i out.img" << std::endl; 
			return 1;
		}

		std::string dataFileName;
		if (!cl.GetOpt("-d", dataFileName)) {
			std::cout << "A data file name must be specified" << std::endl;
			return 1;
		}

		std::string imageFileName;
		if (!cl.GetOpt("-i", imageFileName)) {
			std::cout << "An image file name must be specified" << std::endl;
			return 1;
		}

		ImageWriterPtr writer = ImageBuilder::GetImageBuilder(imageFileName)->GetWriter();

		// limits in .hdr file, but assume size for convenience
		// image is one less in each direction due to lighting calculation and cross product
		writer->Open(imageFileName, 10811,10811, 3,8);  // data is 8 bits/channel RGB
		ImageRow row = writer->GetImageRow();

		unsigned int rowsWritten = 0;
		FileReader dataFile(dataFileName);
		std::unique_ptr<std::vector<float>> rowA(new std::vector<float>);
		std::unique_ptr<std::vector<float>> rowB(new std::vector<float>);
		if (!dataFile.ReadRow(*rowA)) {
			std::cout << "Cannot read row" << std::endl;
			return 1;
		}

		for (unsigned int y=1; y<10812; y++) { // 1 since already read a row
			if (rowsWritten%1000 == 0)
				std::cout << rowsWritten << std::endl;

			if (!dataFile.ReadRow(*rowB)) {
				std::cout << "Cannot read row" << std::endl;
				return 1;
			}

			// use rowA and rowB with cross product/lighting calculation
			// write lit image row
			for (unsigned int x=1; x<10812; x++) {
				if ((*rowA)[x] == -9999)  // no data
					row.SetPixel(x-1, (unsigned char)0,0,0,255);  // -1 due to cross product
				else {
					// compute cross product between x,y,z and x-1,y,z and x,y-1,z
					float v1x = -1, v1y=0, v1z = (*rowB)[x-1]-(*rowB)[x];
					float v2x = 0, v2y=-1, v2z=(*rowA)[x]-(*rowB)[x];
					float vnx = /*v1y*v2z-v1z*v2y;*/ v1z;
					float vny = /*v1z*v2x-v1x*v2z;*/ v2z;
					float vnz = /*v1x*v2y-v1y*v2x;*/ 1;

					// normalize
					float mag = sqrt(vnx*vnx+vny*vny+vnz*vnz);
					vnx/=mag; vny/=mag; vnz/=mag;

					// do lighting calc against vn l dot n
					float lx=-1000.0f, ly=-1000.0f, lz=2000.0f;
					mag = sqrt(lx*lx+ly*ly+lz*lz);
					lx/=mag; ly/=mag; lz/=mag;

					float dot = std::max(lx*vnx+ly*vny+lz*vnz, 0.0f);  // back-facing polygons give a negative dot product
					// dot = std::min(dot + 0.05f, 1.0f); // ambient term  
					unsigned char c = (unsigned char)(255*dot);
					row.SetPixel(x-1, c,c,c,255);  // -1 due to cross product
				}
			}

			if (rowsWritten < writer->GetHeight()) {
				writer->WriteRow(row);
				rowsWritten++;
			}
			rowA.swap(rowB);  // shift down
		}

		std::cout << "Rows written: " << rowsWritten << std::endl;
	} catch (std::exception &e) {
		std::cout << "Error: " << e.what() << std::endl;
	}

	return 0;
}
