#include "Half.h"
#include "ImageBuilder.h"
#include <sstream>


void Half(ImageWriterPtr writer, ImageRow &srcRow, ImageRow &destRow, unsigned int halfSrcWidth, unsigned int destWidthExtra) {
	unsigned short r,g,b,a;
	for (unsigned int x=0; x<halfSrcWidth; x++) {
		srcRow.GetPixel(x*2, r,g,b,a);  // drop every other pixel
		destRow.SetPixel(x, r,g,b,a);
	}
	if (destWidthExtra>0) {
		// odd number of destination pixels in the row, so copy one more
		srcRow.GetPixel(halfSrcWidth, r,g,b,a);
		destRow.SetPixel(halfSrcWidth, r,g,b,a);
	}
	writer->WriteRow(destRow);
}

// reduce image size by half in each dimension, rounding up if the count is odd (20000 -> 10000, 20001 -> 10001, 20002 -> 10001)
void EXT_CLASS_IMAGELIB Half(const std::string &srcFileName, const std::string &destFileName, ProgressFn progress) {
	ImageReaderPtr reader = ImageBuilder::GetImageBuilder(srcFileName)->GetReader();
	ImageWriterPtr writer = ImageBuilder::GetImageBuilder(destFileName)->GetWriter();

	reader->Open(srcFileName);

	unsigned int halfSrcWidth = reader->GetWidth()/2;
	unsigned int halfSrcHeight = reader->GetHeight()/2;
	unsigned int destWidthExtra = reader->GetWidth()%2;
	unsigned int destHeightExtra = reader->GetHeight()%2;

	unsigned int destWidth = halfSrcWidth + destWidthExtra;		// round up if odd
	unsigned int destHeight = halfSrcHeight + destHeightExtra;	// round up if odd

	writer->Open(destFileName, destWidth, destHeight, reader->GetChannelCount(), reader->GetChannelDepth());  // for very large images, problem if go from PNG->JPEG due to limits

	ImageRow srcRow = reader->GetImageRow();
	ImageRow destRow = writer->GetImageRow();

	// loop to src->height/2 and src->width/2 as dest will be 2x this, or 2x this plus one
	for (unsigned int y=0; y<halfSrcHeight; y++) {
		// show progress
		if ((y%1000 == 0) && progress) {
			std::stringstream s;
			s << "    Row " << y << " of " << destHeight;
			progress(s.str());
		}

		// row to keep
		reader->ReadNextRow(srcRow);
		Half(writer, srcRow, destRow, halfSrcWidth, destWidthExtra);

		// row to discard
		reader->ReadNextRow(srcRow);
	}

	// odd number of rows, so copy one more
	if (destHeightExtra>0)
		Half(writer, srcRow, destRow, halfSrcWidth, destWidthExtra);

	if (progress) {
		std::stringstream s;
		s << "    Done " << destHeight;
		progress(s.str());
	}
}
