﻿#define __CL_ENABLE_EXCEPTIONS

#include <iostream>
#include <fstream>
#include <omp.h>
#include <CL/cl.hpp>
#include <iomanip>
#include "PNGWriter.h"

std::string LoadKernelSource(std::string filename) {
	// look in current dir - if missing, look in parent dir - so exe can be in Release subdir of source, for instance
	std::ifstream file(filename);
	if (!file.is_open()) {
		file.open("../" + filename);
		if (!file.is_open())
			throw std::exception("File not found");
	}

	return std::string((std::istreambuf_iterator<char>(file)), std::istreambuf_iterator<char>());
}


cl::size_t<3> range3(size_t a, size_t b, size_t c) {
	cl::size_t<3> range;
	range.push_back(a); range.push_back(b); range.push_back(c);
	return range;
}

typedef struct Matrix33 {
	float c0[3]; float _pad1;
	float c1[3]; float _pad2;
	float c2[3]; float _pad3;
} Matrix33;

std::ostream &operator<<(std::ostream &os, const float x[3]) {
	os << std::setw(6) << x[0] << "," << std::setw(6) << x[1] << "," << std::setw(6) << x[2];
	return os;
}

std::ostream &operator<<(std::ostream &os, const int x[2]) {
	os << std::setw(6) << x[0] << "," << std::setw(6) << x[1];
	return os;
}

std::ostream &operator<<(std::ostream &os, const Matrix33 &m) {
	// column-major order
	for (int r=0; r<3; r++)
		os << "|" << std::setw(6) << m.c0[r] << "," << std::setw(6) << m.c1[r] << "," << std::setw(6) << m.c2[r] << "|" << std::endl;
	return os;
}

#pragma pack(push, 1)
typedef struct {
	// rendering parameters
	float _cameraPosition[3];  float _pad1; // in OpenCL, sizeof(float) = 4, but a float3 has alignment 4*sizeof(float), so need float[4] on the C++ side (sec 6.1.5 of spec)
	float _cameraOrientation[3]; float _pad2;
	float _objectOrientation[3]; float _pad3;
	int _antialiasing;
	int _width;  // overall image dimensions
	int _height;
	float _cameraZoom;
	float _ambient[3]; float _pad4;
	float _lightPos[3]; float _pad5;
	float _lightColor[3]; float _pad6;
	float _maxRayLength;
	float _shadows;
	int _type;
	int _fog;
	float _fogColor[3]; float _padf;
	float _fogMultiplier;
	float _fogPower;

	// Mandelbox parameters
	float _scale;  // mandelbox scale factor
	int _maxIterations;
	int _maxSteps;
	float _boxFoldColor[3];  float _pad7;
	float _sphereFoldColor[3];  float _pad8;
	float _specularColor[3]; float _pad9;
	float _specularExponent;

	// additional parameters for the Mandelbulb 
	float _bailout;
	int _useJulia;
	float _julia_c[3]; float _pad11;
	float _power;

	// set by Initialize
	float _aspectRatio;
	Matrix33 _viewRotation;
	Matrix33 _objRotation;
	float _eye[3]; float _pad10;
	float _sampleStep;
	float _sampleContribution;
} Params;
#pragma pack(pop)


template <typename T>
void Set(T *v, T v0, T v1) {
	v[0] = v0; v[1] = v1;
}

template <typename T>
void Set(T *v, T v0, T v1, T v2) {
	v[0] = v0; v[1] = v1; v[2] = v2;
}



int main(int argc, char *argv[]) {
	cl::Program program;
	std::vector<cl::Device> devices;
	try {
		// get a platform and device
		std::vector<cl::Platform> platforms;
		cl::Platform::get(&platforms);
		if (platforms.size() == 0) {
			std::cout << "OpenCL not available" << std::endl;
			return 1;
		}

		// create context and queue
		cl_context_properties cprops[3] = { CL_CONTEXT_PLATFORM, (cl_context_properties)platforms[0](), 0 };
		cl::Context context = cl::Context(CL_DEVICE_TYPE_GPU, cprops);

		devices = context.getInfo<CL_CONTEXT_DEVICES>();
		if (devices.size() == 0) {
			std::cout << "GPU device not available" << std::endl;
			return 1;
		}

		cl::CommandQueue queue = cl::CommandQueue(context, devices[0]);

		// compile source, get kernel entry point
		std::string source = LoadKernelSource("../../OCL_Mand/Mandelbox.cl");
		cl::Program::Sources sources(1, std::make_pair(source.c_str(), source.size()));
		program = cl::Program(context, sources);
		program.build(devices);
		cl::Kernel initKernel = cl::Kernel(program, "Initialize");
		cl::Kernel renderKernel = cl::Kernel(program, "Mandelbox");

		double startTime = omp_get_wtime();

		unsigned int width = 12800;  // 12800 7200
		unsigned int height = 7200;

		PNGWriter writer;
		writer.Open("image.png", width, height, 3, 8);
		ImageRow row = writer.GetImageRow();

		Params params;

		// Rendering parameters
		Set(params._cameraPosition, 0.0f, -15.0f, 0.0f);   // 0,-5,0 at scale 3 is interesting
		Set(params._cameraOrientation, 0.0f, 0.0f, -90.0f);  // 0 0 -90
		Set(params._objectOrientation, 0.0f, 30.0f, 40.0f);  // Mandelbox: 0 rot(frame) 0  Mandelbulb: 0 38.5 25.8
		params._antialiasing = 1;  // 1=no antialiasing, >1 antialias
		params._width = width;
		params._height = height;
		params._cameraZoom = 0;
		Set(params._ambient, 0.1f, 0.1f, 0.1f);
		// Set(params._lightPos, -10.0f, -40.0f, -10.0f);
		////Set(params._lightPos, -10.0f, -10.0f, 0.0f);  // -10 -10 0
		Set(params._lightPos, -5.0f, -10.0f, -5.0f);  // 0 -10 0 is right behind camera when cameraPos is 0 -6 0
		//Set(params._lightColor, 0.7f, 0.7f, 0.7f);
		Set(params._lightColor, 1.0f, 1.0f, 1.0f);
		params._maxRayLength = 30; // 200  for -3, 20 is too far, 2 shows nothing
		params._shadows = 0.2f;
		Set(params._specularColor, 0.4f, 0.4f, 0.4f);  // 0.4 0.4 0.4
		params._specularExponent = 8.0f;

		// Mandelbox parameters
		params._scale = 2.5;  // -3 is good at 0,-6,0   with +3, 0,-6,0 is inside the object (0,-1,0 is in a curve)
		params._maxIterations = 50;
		params._maxSteps = 200;
		params._type = 1; // 1 = Mandelbox

		params._fog = 1;
		Set(params._fogColor, 1.0f,1.0f,1.0f);
		params._fogMultiplier = 1.5;
		params._fogPower = 1.3f;

		Set(params._boxFoldColor, 1.0f,1.0f,1.0f);  // 0.4 0 0 
		Set(params._sphereFoldColor, 0.0f,1.0f,0.0f);  // 0 .4 0
		/*
		params._type = 2;  // 2 = Mandelbulb
		Set(params._julia_c, 0.0f, 0.0f, 0.0f);  // 1 0 0  0..2 for first param
		params._useJulia = 1;  // false
		params._bailout = 4.0f;
		params._power = 2.0;  // 8.0f
		Set(params._cameraPosition, 0.0f, -4.0f, 0.0f);
		Set(params._lightPos, -10.0f, -40.0f, -10.0f);
		Set(params._objectOrientation, 0.0f, 0.0f, 0.0f);
		Set(params._cameraOrientation, 0.0f, 0.0f, -90.0f);  // 0 0 -90
		*/

		// Initialize parameters
		cl::Buffer initParamsBuffer = cl::Buffer(context, CL_MEM_READ_WRITE | CL_MEM_COPY_HOST_PTR, sizeof(Params), &params);
		initKernel.setArg(0, initParamsBuffer);
		queue.enqueueNDRangeKernel(initKernel, cl::NDRange(), cl::NDRange(1), cl::NDRange());  // just initialize the one structure

		int imageBufferSize = 4*width;  // RGBA
		cl::Buffer imageBuffer = cl::Buffer(context, CL_MEM_WRITE_ONLY, imageBufferSize);  // 1 byte each, RGBA

		renderKernel.setArg(0, initParamsBuffer);
		renderKernel.setArg(1, imageBuffer);
		for (unsigned int currentRow = 0; currentRow < height; currentRow++) {
			if (currentRow%100 == 0)
				std::cout << currentRow << std::endl;

			renderKernel.setArg(2, currentRow);
			queue.enqueueNDRangeKernel(renderKernel, cl::NDRange(), cl::NDRange(width), cl::NDRange());  // calculate just one row of the image
			queue.finish();

			// write this out to a file
			unsigned char *imagePtr = (unsigned char *)queue.enqueueMapBuffer(imageBuffer, CL_TRUE, CL_MAP_READ, 0, imageBufferSize);
			for (unsigned int x=0; x<width; x++)
				row.SetPixel(x, *(imagePtr+4*x), *(imagePtr+4*x+1), *(imagePtr+4*x+2), *(imagePtr+4*x+3));
			writer.WriteRow(row);
		}

		double endTime = omp_get_wtime()-startTime;
		std::cout << "Calculation time: " << endTime << std::endl;

	} catch (cl::Error &err) {
		std::cout << "Error: " << err.what() << "(" << err.err() << ")" << std:: endl;
		// if it was a compilation error
		if (err.err() == CL_BUILD_PROGRAM_FAILURE)
			std::cout << program.getBuildInfo<CL_PROGRAM_BUILD_LOG>(devices[0]) << std::endl;
	} catch (std::exception &e) {
		std::cout << "Error: " << e.what() << std::endl;
	}

	return 0;
}

