#include "JPEGReader.h"
#include <stdlib.h>

void error_exit(j_common_ptr cinfo) { exit(1); }

void JPEGReader::ErrorHandler(j_common_ptr cinfo) {
	j_common_ptr mycinfo = reinterpret_cast<j_common_ptr>(cinfo);

	// ONLY DIFFERENCE BETWEEN READER AND WRITER ERROR HANDLING - don't care if not everything was read (especially if just wanting to get size info)
	// could use client_data to distinguish this condition and use same function for error handling
	if (mycinfo->err->msg_code == JERR_TOO_LITTLE_DATA)
		return; // don't bother throwing this one as an error

	// create the message
	char buffer[JMSG_LENGTH_MAX];
	(*mycinfo->err->format_message) (mycinfo, buffer);

	// disconnect the error handler so an exception on object cleanup won't disrupt processing of this exception - process is expected to abort()
	mycinfo->err->error_exit = error_exit;

	// Let the memory manager delete any temp files before we die
	jpeg_destroy(mycinfo);
	throw ImageException(buffer);
}

// libjpeg.txt

void JPEGReader::Open(const std::string &fileName) {
	Close();

	_cinfo.err = jpeg_std_error(&_jerr);
	_jerr.error_exit = JPEGReader::ErrorHandler;
	jpeg_create_decompress(&_cinfo);

	_fp = fopen(fileName.c_str(), "rb");
	if (!_fp) {
		jpeg_destroy_decompress(&_cinfo);
		throw ImageException(std::string("Could not open ") + fileName + std::string(" for reading"));
	}

	jpeg_stdio_src(&_cinfo, _fp);
	jpeg_read_header(&_cinfo, TRUE);

	jpeg_start_decompress(&_cinfo);
	if ( (_cinfo.out_color_components!=3) && (_cinfo.out_color_components!=4) )
		throw ImageException("Only RGB and RGBA color channel images are supported");
	_channelCount = _cinfo.out_color_components;
	_channelDepth = 8;

	if ((_cinfo.output_width==0) || (_cinfo.output_height==0)) {
		jpeg_finish_decompress(&_cinfo);
		jpeg_destroy_decompress(&_cinfo);
		fclose(_fp);
		_fp=0;
		throw ImageException("Image is of zero dimension");
	}

	_width = _cinfo.output_width;
	_height = _cinfo.output_height;
}


void JPEGReader::Close() {
	if (_fp) {
		jpeg_finish_decompress(&_cinfo);
		jpeg_destroy_decompress(&_cinfo);
		fclose(_fp);
	}
	_fp = 0;
}


void JPEGReader::ReadNextRow(ImageRow &row) {
	if ((row.GetWidth() != _width) || (row.GetChannelCount() != _channelCount) || (row.GetChannelDepth() != _channelDepth))
		row = GetImageRow();  // only allocate a new row buffer if the existing one isn't compatible
	unsigned char *p = row.Get();
	jpeg_read_scanlines(&_cinfo, &p, 1);
}


