#include "Tile.h"
#include "ImageBuilder.h"
#include <iostream>
#include <vector>
#include <sstream>
#include <algorithm>
#include <math.h>
#include <fstream>
#include <iomanip>
#include <iterator>
#include <fstream>
#include "FileManagement.h"
#include "Crop.h"
#include "Half.h"


std::string operator+(const std::string &str, unsigned int i) {
	std::stringstream s;
	s << str << i;
	return s.str();
}


std::string GetLevelFilename(const std::string &fileName, unsigned int level) {
	std::stringstream s;
	s << FileManagement::GetBasename(fileName) << "_" << level << ImageBuilder::GetImageBuilder(fileName)->GetExtension();
	return s.str();
}


void GetTileParams(const std::string &fileName, unsigned int &width, unsigned int &height, unsigned int &maximumLevel) {
	// get the width and height (in pixels) of the image to zoom into
	ImageReaderPtr reader = ImageBuilder::GetImageBuilder(fileName)->GetReader();
	reader->Open(fileName);
	width = reader->GetWidth();
	height = reader->GetHeight();
	reader->Close();

	// determine the number of levels needed for the zoom
	maximumLevel = (unsigned int)ceil(log((double)std::max(width, height))/log(2.0));
}


void BuildLevels(const std::string &fileName, unsigned int maximumLevel, ProgressFn progress) {
	if (progress)
		progress("Build levels...");

	// the deepest image level is the same as the main image - just copy the existing one
	if (progress)
		progress("  " + maximumLevel);
	FileManagement::Copy(fileName, GetLevelFilename(fileName, maximumLevel));

	// iterate over the remaining levels (SHOULD DELETE THESE FILES WHEN DONE)
    for (unsigned int levelR = 0; levelR <=maximumLevel-1; levelR++) {
		unsigned int level = maximumLevel-1-levelR;  // to avoid problem with decrementing for loop and unsigned var
		if (progress)
			progress("  " + level);
		try {
			// reduce the previous level by half
			Half(GetLevelFilename(fileName, level+1), GetLevelFilename(fileName, level), progress);  
		} catch (std::exception &e) {
			throw ImageException(std::string("Level: ") + level + ": " + e.what());
		}
	}
}


void BuildTiles(const std::string &fileName, const std::string &tileExt, unsigned int maximumLevel, unsigned int width, unsigned int height,
	unsigned int tileSize, unsigned int overlap, ProgressFn progress) {
	if (progress)
		progress("Build tiles...");

	std::string mainDir = FileManagement::GetBasename(fileName) + "_files";
	FileManagement::CreateDir(mainDir);

    for (unsigned int level = 0; level <= maximumLevel; level++) {
		std::string levelDir = mainDir + FileManagement::PathSep() + level;
		FileManagement::CreateDir(levelDir);

		// don't cache more rows than needed for a single tile row (tile height plus overlap)
		RowCache rowCache(GetLevelFilename(fileName, level), tileSize+2*overlap);

		// get the width and height (in pixels) of the current level being processed
		double scale = 1.0 / (1 << (maximumLevel - level));
		unsigned int levelWidth = (unsigned int)(ceil(width*scale));
		unsigned int levelHeight = (unsigned int)(ceil(height*scale));

		// get the number of tiles needed to cover the level dimensions
		unsigned int numTileColumns = (unsigned int)(ceil(levelWidth / (double)tileSize));
		unsigned int numTileRows = (unsigned int)(ceil(levelHeight / (double)tileSize));

		for (unsigned int row = 0; row < numTileRows; row++)  // be sure to use the cropper like this for best efficiency - y as the outer loop
			for (unsigned int column = 0; column < numTileColumns; column++) {

				// tile starting pixel
				unsigned int tileX = (column == 0) ? 0 : (tileSize * column - overlap);
				unsigned int tileY = (row == 0) ? 0 : (tileSize * row - overlap);

				// tile dimensions
				unsigned int tileWidth = tileSize + ((column == 0) ? 1 : 2) * overlap;
				unsigned int tileHeight = tileSize + ((row == 0) ? 1 : 2) * overlap;

				// ensure the tile doesn't go past the edge of the level
				tileWidth = std::min(tileWidth, levelWidth - tileX);
				tileHeight = std::min(tileHeight, levelHeight - tileY);

				if (progress)
					progress(std::string("  ") + level + ": " + column + "," + row);

				// crop the tile from the level image and save it
				std::string tilePath = levelDir + FileManagement::PathSep() + column + "_" + row + ((tileExt.size()==0)?ImageBuilder::GetImageBuilder(fileName)->GetExtension() : tileExt);
				Crop(rowCache, tileX,tileY, tileWidth,tileHeight, tilePath);
            }
    }
}


void BuildDZI(const std::string &fileName, const std::string &tileExt, unsigned int width, unsigned int height, unsigned int tileSize, unsigned int overlap) {
    // create the dzi file
	std::stringstream dzi;
	dzi << "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n" <<
		"<Image TileSize=\"" << tileSize << "\" Overlap=\"" << overlap << "\" Format=\"" + 
		((tileExt.size()==0)?ImageBuilder::GetImageBuilder(fileName)->GetExtension() : tileExt).substr(1) // drop the leading .
		+ "\"\n" <<   
		"       xmlns=\"http://schemas.microsoft.com/deepzoom/2008\">\n" <<
		"       <Size Width=\"" << width << "\" Height=\"" << height << "\" />\n" <<
		"</Image>\n";

	std::ofstream dziFile((FileManagement::GetBasename(fileName) + ".dzi").c_str());
	dziFile << dzi.str();
	dziFile.close();
}


void Tile(const std::string &fileName, const std::string &tileExt, unsigned int tileSize, unsigned int overlap, ProgressFn progress) {
	unsigned int width, height, maximumLevel;
	GetTileParams(fileName, width, height, maximumLevel);
	BuildLevels(fileName, maximumLevel, progress);
	BuildTiles(fileName, tileExt, maximumLevel, width, height, tileSize, overlap, progress);
	BuildDZI(fileName, tileExt, width, height, tileSize, overlap);
}

