#include "DDSPublisherFacade.h"
#include <dds/DCPS/Marked_Default_Qos.h>
#include <dds/DCPS/Qos_Helper.h>
#include <dds/DCPS/Service_Participant.h>

DDSPublisherFacade::DDSPublisherFacade(
	DDS::DomainParticipantFactory_ptr factory,
	DDS::DomainId_t domain, 
	const char* topicName)
	:factory_ (factory)
{
	createParticipant(domain);
	createTopic(topicName);
	createPublisher();
	createDataWriter();	
}


DDSPublisherFacade::~DDSPublisherFacade()
{

}

void  DDSPublisherFacade::cleanup()
{
		// Delete any topics, publishers and subscribers owned by participant
	participant_->delete_contained_entities();

	// Delete participant itself
	factory_->delete_participant(participant_);

	// Shut down info repo connection
	TheServiceParticipant->shutdown();

}


void DDSPublisherFacade::createParticipant(
	DDS::DomainId_t domain)
{
	participant_ =
		factory_->create_participant(domain, 
								     PARTICIPANT_QOS_DEFAULT,
									 0,  // no listener
		                             OpenDDS::DCPS::DEFAULT_STATUS_MASK);

	// Check for failure
	if (!participant_) {
		throw std::string("failed to create domain participant");
	}
}


DDS::Topic_var DDSPublisherFacade::createTopic(
	const char* topicName)
{
	// Register TypeSupport (Messenger::Message)
	Messenger::MessageTypeSupport_var ts =
		new Messenger::MessageTypeSupportImpl;

	if (ts->register_type(participant_, "") != DDS::RETCODE_OK) {
		throw std::string("failed to register type support");
	}

	// Create Topic 
	CORBA::String_var type_name = ts->get_type_name();
	topic_ =
		participant_->create_topic(topicName,
		type_name,
		TOPIC_QOS_DEFAULT,
		0,
		OpenDDS::DCPS::DEFAULT_STATUS_MASK);

	// Check for failure
	if (!topic_) {
		throw std::string("failed to create topic");
	}
	return topic_;
}

DDS::Publisher_var
	DDSPublisherFacade::createPublisher()
{
	// Create Publisher
	publisher_ =
		participant_->create_publisher(
		PUBLISHER_QOS_DEFAULT,
		0,
		OpenDDS::DCPS::DEFAULT_STATUS_MASK);

	// Check for failure
	if (!publisher_) {
		throw std::string("failed to create publisher");
	}
	return publisher_;
}


Messenger::MessageDataWriter_var
	DDSPublisherFacade::createDataWriter()
{

	::DDS::DataWriterQos qos;
	publisher_->get_default_datawriter_qos(qos);

	//Add liveliness qos policy
	qos.liveliness.kind = DDS::AUTOMATIC_LIVELINESS_QOS;

	//Set the duration
	DDS::Duration_t liveliness_duration;
	liveliness_duration.sec = 2;
	liveliness_duration.nanosec = 0;
	qos.liveliness.lease_duration = liveliness_duration;

	// Create DataWriter
	DDS::DataWriter_var writer =
		publisher_->create_datawriter(topic_,
		qos,
		0,
		OpenDDS::DCPS::DEFAULT_STATUS_MASK);

	// Safe cast to type-specific data writer
	messageWriter_ =
		Messenger::MessageDataWriter::_narrow(writer);

	// Check for failure
	if (!messageWriter_) {
		throw std::string("failed to narrow data writer");
	}

	return messageWriter_;
}


DDS::ReturnCode_t DDSPublisherFacade::write(
	Messenger::Message& message)
{
	return messageWriter_->write(message, DDS::HANDLE_NIL);
}

Messenger::MessageDataWriter_var
DDSPublisherFacade::getDataWriter()
{
	return messageWriter_;
}
