#ifndef MESSAGEENGINECOMPONENT_CPP
#define MESSAGEENGINECOMPONENT_CPP
#include <dds/DCPS/Service_Participant.h>		// OpenDDS environment support
#include "MessageEngineComponent.h"
#include "MessageEngineTypeSupportImpl.h"		// MessageEngine IDL type support
#include "MessageEngineDDSDataListenerImpl.h"	// OpenDDS listener for Enterprise messages
#include "ddsutil/DDSFilteredSubscriberFacade.h"
#include "ddsutil/DDSSubscriberFacade.h"		// OpenDDS Subscriber support
#include <Swiften/Swiften.h>					// XMPP library support
#endif

#include <iostream>
#include <boost/bind.hpp>



using namespace Swift;
using namespace boost;

MessageEngineComponent::MessageEngineComponent(bool verbose): present_(false), joined_(false), localRoom_("default") ,verbose_(false), trace_(false){

		verbose_=verbose;

}


MessageEngineComponent::~MessageEngineComponent() {
      delete tracer;
	  delete XMPPclient_;
}
  

void MessageEngineComponent::handlePresenceReceived(Presence::ref presence) {
	// Automatically approve subscription requests
	if (presence->getType() == Presence::Subscribe) {
		Presence::ref response = Presence::create();
		response->setTo(presence->getFrom());
		response->setType(Presence::Subscribed);
		XMPPclient_->sendPresence(response);
	}
}

void MessageEngineComponent::handleConnected() {

	std::cout << "Connected to XMPP server..." << std::endl;
	// Request the roster
	GetRosterRequest::ref rosterRequest = 
		GetRosterRequest::create(XMPPclient_->getIQRouter());
	rosterRequest->onResponse.connect(
		bind(&MessageEngineComponent::handleRosterReceived, this, _2));
	rosterRequest->send();

	// Create MUC Session

	Swift::MUCManager * localMUCManager_ = XMPPclient_->getMUCManager();
	XMPPMUChandle_ = localMUCManager_->createMUC(localRoom_);
	XMPPMUChandle_->onJoinComplete.connect(boost::bind(&MessageEngineComponent::handleJoinComplete,this,_1));

	// Don't want to join the room until Devices are present.
	XMPPMUChandle_->joinAs("MessageEngine");
//	response->setType(Swift::Presence::Available);
	joined_ = true;

}

void MessageEngineComponent::handleDataRead(const SafeByteArray& data) {

	if (trace_) {
		Swift::XMLBeautifier beautifier(true,true);
		std::cout << beautifier.beautify(byteArrayToString(ByteArray(data.begin(), data.end()))) << std::endl;
	}
}


void MessageEngineComponent::handleDataWritten(const SafeByteArray& data) {

	if (trace_) {
		Swift::XMLBeautifier beautifier(true,true);
		std::cout << std::cout << beautifier.beautify(byteArrayToString(ByteArray(data.begin(), data.end()))) << std::endl;
	}
}

void MessageEngineComponent::handleJoinComplete(const std::string& nick) {

	 	Swift::Presence::ref response = Swift::Presence::create();
		response->setType(Swift::Presence::Available);
		XMPPclient_->sendPresence(response);

	if (verbose_) {
		std::cout << "Join complete for:" << nick << std::endl;
	}


}

// The input parameter to this method is a Boost smart pointer to a 
// Swift Message data type that has been received

void MessageEngineComponent::handleMessageReceived(Swift::Message::ref message) {

	// Create a DDS message to load the XMPP content into
       // This data type is defined in MessageEngine.idl
	Messenger::Message ddsMsg;

      // An XMPP message isnt required to have a subject or subject id
	if(message->hasSubject()) {
		ddsMsg.subject_id = atol(message->getID().c_str());
		ddsMsg.subject    = message->getSubject().c_str();
	}

      // Now populate the remainder of the DDS message with the XMPP content.
	ddsMsg.from       = message->getFrom().toString().c_str();
	ddsMsg.text       = message->getBody().c_str();
	ddsMsg.count      = 1; // simple number of messages in this transmission


       // Now publish the newly created DDS message
	publisher_->write(ddsMsg);

}


void MessageEngineComponent::handleRosterReceived(ErrorPayload::ref error) {
	if (error) {	

		std::cout << "Error receiving roster. Continuing anyway." << std::endl;
	}

	// Send initial available presence
	XMPPclient_->sendPresence(Presence::create("Send me a message"));
}

int  MessageEngineComponent::initDDS(int  argc, char * argv[]) {
	//Create OpenDDS environment to connect to enterprise messages.

	// Initialize DomainParticipantFactory, handling command line args
	DDS::DomainParticipantFactory_var dpf =
		TheParticipantFactoryWithArgs(argc, argv);

	const DDS::DomainId_t domain = 42;

	try {

		const char* xmppTopic = "CommandMessages";

		publisher_ = new DDSPublisherFacade(dpf, domain, xmppTopic);
	}   catch (std::exception & e)
	{
		std::cout << "Exception: " << e.what() << std::endl;
		return -1;
	}   catch (CORBA::Exception& e) 
	{
		std::cout << "Exception: " << e << std::endl;
		return -1;
	}

	if (verbose_){
		std::cout << "Connected to OpenDDS Discovery..." << std::endl;
	}

	//Initiate a Subscriber so we can receive messages from Enterprise Messaging clients
	try {

		const char* devTopic = "DeviceMessages";

		/* Instantiate a DataReaderListener to use for the subscriber */
		DDS::DataReaderListener_var  devMessagesListener(new MessageEngineDDSDataListenerImpl(XMPPclient_, localRoom_, verbose_));

		subscriber_ = new DDSSubscriberFacade(dpf, domain, devTopic, devMessagesListener);
	}   catch (std::exception & e)
	{
		std::cout << "Exception: " << e.what() << std::endl;
		return -1;
	}   catch (CORBA::Exception& e) 
	{
		std::cout << "Exception: " << e << std::endl;
		return -1;
	}

	return 0;
}

int MessageEngineComponent::initXMPP(SimpleEventLoop* eventLoop, 
	NetworkFactories* networkFactories,
	std::string uid,
	std::string pwd,
	std::string room,
	bool verbose,
	bool trace) {

		localRoom_ = room;
		trace_ = trace;
		XMPPclient_ = new Client(uid, pwd, networkFactories);
		XMPPclient_->setAlwaysTrustCertificates();
		XMPPclient_->onConnected.connect(boost::bind(&MessageEngineComponent::handleConnected,this));
		XMPPclient_->onDataRead.connect(boost::bind(&MessageEngineComponent::handleDataRead,this,_1));
		XMPPclient_->onDataWritten.connect(boost::bind(&MessageEngineComponent::handleDataWritten,this,_1));
		XMPPclient_->onMessageReceived.connect(boost::bind(&MessageEngineComponent::handleMessageReceived,this, _1));
		XMPPclient_->onPresenceReceived.connect(boost::bind(&MessageEngineComponent::handlePresenceReceived,this, _1));

		//if(trace) {
		//	tracer = new ClientXMLTracer(XMPPclient_,false);
		//}

		XMPPclient_->connect(); // Make this last 


		if (!(XMPPclient_->isActive())) {
			std::cout << "XMPP client did not connect properly...check uid/pw." << std::endl;
			return -1;
		}

		return 0;

}

void MessageEngineComponent::setPresence(bool presence) {
	if(presence && !present_) {
		this->present_=true;
		//announce presence
		// Request the roster
		XMPPclient_->sendPresence(Presence::create("Send me a message"));
	} else if(!presence && present_) {
		Presence::ref myPresence = Presence::create();
		myPresence->setType(Presence::Unavailable);
		XMPPclient_->sendPresence(myPresence);
	}

}
