/*
* $Id: DeviceSimulation.cpp $
*
*
* Distributed under the OpenDDS License.
* See: http://www.opendds.org/license.html
*/

#include "DeviceSimulation.h"
#include "Boilerplate.h"
#include "DDSPublisherFacade.h"
#include "MessageEngineTypeSupportImpl.h"
#include "ace/Thread_Manager.h"
#include <string>

using namespace examples::boilerplate;

// Initialize the worker thread device pointer
DeviceSimulation* DeviceSimulation::deviceWorker_ = 0;

DeviceSimulation::DeviceSimulation(
	const char* device_id, 
	DDSPublisherFacade* publisher,
	int publication_period) 
	: sample_count(0), aliveCount_(0), pressure_(50.0), temperature_(200), 
	  publisher_(publisher), deviceId_(device_id), publicationPeriod_(publication_period)
{
	message_.subject_id = 0;
	message_.from       = device_id;
	message_.subject    = "";
	message_.text       = "";
	message_.count      = 0;
}

DeviceSimulation::~DeviceSimulation()
{
	
}

void DeviceSimulation::on_data_available(
	DDS::DataReader_ptr reader)
	ACE_THROW_SPEC((CORBA::SystemException))
{
	// Safely downcast data reader to type-specific data reader
	Messenger::MessageDataReader_var reader_i = narrow(reader);

	Messenger::Message msg;
	DDS::SampleInfo info;

	// Remove (take) the next sample from the data reader
	DDS::ReturnCode_t error = reader_i->take_next_sample(msg, info);

	// Make sure take was successful
	if (error == DDS::RETCODE_OK) {
		// Make sure this is not a sample dispose message
		if (info.valid_data) {
			++sample_count;
			std::cout << "Msg: subject    " << msg.subject.in() << std::endl
				<< "     subject_id " << msg.subject_id   << std::endl
				<< "     from       " << msg.from.in()    << std::endl
				<< "     count      " << msg.count        << std::endl
				<< "     text       " << msg.text.in()    << std::endl;

			// Parse the message to and process command
			this->parseMessage(msg);
		}
	} else {
		ACE_ERROR((LM_ERROR,
			ACE_TEXT("ERROR: %N:%l: on_data_available() -")
			ACE_TEXT(" take_next_sample failed!\n")));
	}
}


void DeviceSimulation::on_liveliness_changed(DDS::DataReader*,
	const DDS::LivelinessChangedStatus& status)
{
	std::cout << "Liveliness: alive = " << status.alive_count << " not_alive = " 
		<< status.not_alive_count << " alive change = " << status.alive_count_change 
		<< " not alive count change = " << status.not_alive_count << std::endl;

	aliveCount_ = status.alive_count;
}

// Parse the message received and if it contains a command for this device, execute it
bool DeviceSimulation::parseMessage(
	const Messenger::Message& message) 
{
	std::string message_text = message.text;
	std::string message_source = message.from;

	// Variables will be assigned by the parseCommand functino
	std::string command_name;
	float command_arg;

	// If it is a command message with one argument it will have three keywords starting with the string
	// "cmd: device_id->command_name = arg
	if (parseCommand(message, command_name, command_arg))
	{
		return executeCommandOneArg(message_source.c_str(), command_name.c_str(), command_arg);
	}

	// If it is a command message with no argument it will have two keywords starting with the string
	// "cmd:" followed by the command name and an argument
	if (parseCommand(message, command_name))
	{
		return executeCommandNoArgs(message_source.c_str(), command_name.c_str());
	}

	// If no command was processed, return failure.

	return false;
}


// Parse the command to determine if it is a command with a float argument. If so, return true 
// and set the command_name to the command that was received and assign the argument to the 
// received value
bool DeviceSimulation::parseCommand(
	const Messenger::Message& message, 
	std::string& command_name, 
	float& float_arg) 
{
	// If it is a command message with one argument it will have three keywords starting with the string
	// "cmd: device_id->command_name arg
	char command_filter[80];
	char received_command[40];
	std::string message_text = message.text;

	sprintf(command_filter, "cmd: %s->%%s %%f", deviceId_.c_str());
	if ((::sscanf (message_text.c_str(), command_filter, received_command, &float_arg) == 2))
	{
		command_name = received_command;
		return true;
	}
	else
		return false;
}

// Parse the command to see if it is a command without arguments and matches one of the
// known command names
bool DeviceSimulation::parseCommand(
	const Messenger::Message& message, 
	std::string& command_name) 
{
	char command_filter[80];
	char received_command[40];
	std::string message_text = message.text;

	
	// If it is a command message with one argument it will have a command after the arrow 
	// "cmd: device_id->command_name" or "cmd: who"
	sprintf(command_filter, "cmd: %s->%%s", deviceId_.c_str());
	if (::sscanf (message_text.c_str(), command_filter, received_command) == 1)
	{
		command_name = received_command;
		return true;
	}
	else /* Check if it is a global command. e.g.,  "cmd: who" */
	{
		sprintf(command_filter, "cmd: %%s");
		if (::sscanf (message_text.c_str(), command_filter, received_command) == 1)
		{
			command_name = received_command;
			return true;
		}
	}
	// No valid command received
	return false; 
}

bool  DeviceSimulation::executeCommandNoArgs(
	const char* command_source, 
	const char* command_name)
{
	if(::strcmp(command_name, "get_pressure") == 0)
	{
		// Handle the message
		char pressure_str[20];
		sprintf(pressure_str, "%f", pressure_);

		DDS::ReturnCode_t status = sendCommandResponse(command_source, command_name, pressure_str);
		if(status == DDS::RETCODE_OK)
			return true;
		else
			return false;
	} 
	else if(::strcmp(command_name, "who") == 0)
	{
		DDS::ReturnCode_t status = sendCommandResponse(command_source, command_name, deviceId_.c_str());
		if(status == DDS::RETCODE_OK)
			return true;
		else
			return false;
	}
	else /* No valid command name received */
	{
		// Attempt to send a rejection back to the sender
		rejectCommand(command_source, command_name, "Invalid command name");
		return false;
	}
}

bool  DeviceSimulation::executeCommandOneArg(
	const char* command_source, 
	const char* command_name, float command_arg)
{
	if(::strcmp(command_name, "set_pressure") == 0)
	{
		// Handle the message
		pressure_ = command_arg;

		// Respond with acknowledgement
		char command_and_arg[40];
		::sprintf(command_and_arg, "%s %f", command_name, command_arg);
		DDS::ReturnCode_t status = acknowledgeCommand(command_source, command_and_arg);
		if( status = DDS::RETCODE_OK)
			return true;
		else
			return false;
	}
	else 
	{
		// No valid command name received. Attempt to send rejection message back
		rejectCommand(command_source, command_name, "Invalid command name");
		return false;
	}
}

DDS::ReturnCode_t DeviceSimulation::acknowledgeCommand(
	const char* command_source, 
	const char* command_name)
{
	return sendMessage(command_source, "Acknowledged", command_name);
}

DDS::ReturnCode_t DeviceSimulation::rejectCommand(
	const char* command_source,
	const char* command_name,
	const char* reason)
{
	std::string reason_message("Rejected: ");
	reason_message += reason;
	return sendMessage(command_source, reason_message.c_str(), command_name);
}

DDS::ReturnCode_t DeviceSimulation::sendCommandResponse(
	const char* command_source,
	const char* command_name,
	const char* command_response)
{
	return sendMessage(command_source, command_name, command_response);
}

DDS::ReturnCode_t DeviceSimulation::sendMessage(
	const char* source,
	const char* subject,
	const char* text)
{ 
	message_.subject = subject;
	message_.from = deviceId_.c_str();
	message_.text = text;

	DDS::ReturnCode_t status = publisher_->write(message_);

	++message_.count;
	++message_.subject_id;

	return status;
}

static void *
device_worker (DeviceSimulation* the_device)
{
	// Set the device up for the worker thread
	DeviceSimulation::deviceWorker_ = the_device;

	bool done = false;
  while(!done) {
		if (ACE_Thread_Manager::instance ()->testcancel (ACE_Thread::self ()) != 0)
		{
			ACE_DEBUG ((LM_DEBUG,
						"(%t) has been cancelled "));
			  break;
		}
		
		if(DeviceSimulation::deviceWorker_ != 0)
		{	
			DeviceSimulation::deviceWorker_->sendStatus();
			ACE_OS::sleep(DeviceSimulation::deviceWorker_->getPublicationPeriod());
		}
		else
		{
			// If the device is gone, we are through
			done = true;
		}
  }	
  // Destructor removes thread from Thread_Manager.
  return 0;
}

DDS::ReturnCode_t DeviceSimulation::sendStatus()
{
	// Simulate temperature
	updateTemperature();

	char temp_string[80];
	::sprintf(temp_string, "Temperature = %f, Pressure = %f", temperature_, pressure_);

	return sendMessage(deviceId_.c_str(), "Sensor Publication", temp_string);
}

void DeviceSimulation::updateTemperature()
{
	if(pressure_ > 80.0) 
	{
		temperature_ += (pressure_/80.0F) * 1.34F;
	}
	else
	{
		if(temperature_ > 200) 
		{
			temperature_ -= 2.47F;
		}
	}
}
void DeviceSimulation::run()
{
	ACE_Thread_Manager *thr_mgr = ACE_Thread_Manager::instance ();

	threadGroup_ =thr_mgr->spawn(ACE_THR_FUNC(device_worker), 
								 reinterpret_cast<void *> (this));

}

void DeviceSimulation::stop()
{
	ACE_Thread_Manager *thr_mgr = ACE_Thread_Manager::instance ();
	DeviceSimulation::deviceWorker_ = 0;
	thr_mgr->cancel_grp(threadGroup_);
}
