#include "DDSFilteredSubscriberFacade.h"
#include <dds/DCPS/Marked_Default_Qos.h>
#include <dds/DCPS/Qos_Helper.h>
#include <dds/DCPS/Service_Participant.h>

DDSFilteredSubscriberFacade::DDSFilteredSubscriberFacade(
	DDS::DomainParticipantFactory_ptr factory,
	DDS::DomainId_t domain, 
	const char* topicName,
	const char* topicFilter,
	DDS::DataReaderListener_var listener)
	: factory_(factory), listener_(listener), dataReader_(0), participant_(0),
	 topic_(0), subscriber_(0)
{
	createParticipant(domain);
	createTopic(topicName, topicFilter);
	createSubscriber();
	createDataReader();
}


DDSFilteredSubscriberFacade::~DDSFilteredSubscriberFacade()
{
	cleanup();
}

void DDSFilteredSubscriberFacade::cleanup()
{
	// Delete any topics, publishers and subscribers owned by participant
	participant_->delete_contained_entities();
	// Delete participant itself
	factory_->delete_participant(participant_);
	
	// Shut down info repo connection
//	TheServiceParticipant->shutdown();

}

void 
DDSFilteredSubscriberFacade::createParticipant(
	DDS::DomainId_t domain)
{
	participant_ =
		factory_->create_participant(domain, // made-up domain ID
		PARTICIPANT_QOS_DEFAULT,
		0,  // no listener
		OpenDDS::DCPS::DEFAULT_STATUS_MASK);

	// Check for failure
	if (!participant_) {
		throw std::string("failed to create domain participant");
	}
}


DDS::Topic_var 
DDSFilteredSubscriberFacade::createTopic(
	const char* topicName,
	const char* topicFilter)
{

	// Register TypeSupport (Messenger::Message)
	Messenger::MessageTypeSupport_var ts =
		new Messenger::MessageTypeSupportImpl;

	// Using a null string for the type name causes the MessageTypeSupport 
	// interface repository identifier to be used as the type name
	if (ts->register_type(participant_, "") != DDS::RETCODE_OK) {
		throw std::string("failed to register type support");
	}

	// Create Topic 
	CORBA::String_var type_name = ts->get_type_name();
	topic_ =
		participant_->create_topic(topicName,
		type_name,
		TOPIC_QOS_DEFAULT,
		0,
		OpenDDS::DCPS::DEFAULT_STATUS_MASK);
	// Check for failure
	if (!topic_) {
		throw std::string("failed to create topic");
	}

	filteredTopic_ =
		participant_->create_contentfilteredtopic("Filtered Message",
			topic_,
			topicFilter,
			DDS::StringSeq() );
	// Check for failure
	if (!filteredTopic_) {
		throw std::string("failed to create topic");
	}
	return topic_;
}



DDS::Subscriber_var
DDSFilteredSubscriberFacade::createSubscriber()
{
	// Create Subscriber
	subscriber_ =
		participant_->create_subscriber(
		SUBSCRIBER_QOS_DEFAULT,
		0,
		OpenDDS::DCPS::DEFAULT_STATUS_MASK);

	// Check for failure
	if (!subscriber_) {
		throw std::string("failed to create subscriber");
	}

	return subscriber_;
}

DDS::DataReader_var
DDSFilteredSubscriberFacade::createDataReader()
{
	::DDS::DataReaderQos qos;
	subscriber_->get_default_datareader_qos(qos);

	//Add liveliness qos policy
	qos.liveliness.kind = DDS::AUTOMATIC_LIVELINESS_QOS;

	//Set the duration
	DDS::Duration_t liveliness_duration;
	liveliness_duration.sec = 2;
	liveliness_duration.nanosec = 0;
	qos.liveliness.lease_duration = liveliness_duration;

	// Create DataReader
	dataReader_ = 
		subscriber_->create_datareader(filteredTopic_,
		qos,
		listener_,
		OpenDDS::DCPS::DEFAULT_STATUS_MASK);

	// Check for failure
	if (!dataReader_) {
		throw std::string("failed to create data reader");
	}

	return dataReader_;
}


DDS::DataReader_var
DDSFilteredSubscriberFacade::getDataReader() const
{
	return dataReader_;
}
