/*
 * Copyright 2009 Andrew Prunicki <prunand@iit.edu,prunicki@ociweb.com>
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.ociweb.jnb.loadrunner;

import java.util.List;

import org.apache.thrift.TException;
import org.apache.thrift.protocol.TBinaryProtocol;
import org.apache.thrift.protocol.TCompactProtocol;
import org.apache.thrift.protocol.TProtocol;
import org.apache.thrift.transport.TFramedTransport;
import org.apache.thrift.transport.TSocket;
import org.apache.thrift.transport.TTransport;

import com.ociweb.jnb.thrift.ConversionHelper;
import com.ociweb.jnb.thrift.ThriftClient;
import com.ociweb.jnb.thrift.ThriftServer;
import com.ociweb.jnb.thrift.gen.Course;
import com.ociweb.jnb.thrift.gen.CourseNotFoundException;
import com.ociweb.jnb.thrift.gen.CourseService;
import com.ociweb.jnb.thrift.gen.UnacceptableCourseException;

public class ThriftLoadRunnerAdapter implements LoadRunner {
    private TSocket socket;
    private TTransport transport;
    private TProtocol protocol;
    private CourseService.Client client;

    public ThriftLoadRunnerAdapter(final String protocolFlag, final String hostName) {
        socket = new TSocket(hostName, ThriftServer.PORT);
        socket.setTimeout(ThriftClient.SOCKET_TIMEOUT);
        transport = new TFramedTransport(socket);
        if (ThriftServer.PROTOCOL_FLAG_BINARY.equals(protocolFlag)) {
            protocol = new TBinaryProtocol(transport);
        } else {
            protocol = new TCompactProtocol(transport);
        }
        client = new CourseService.Client(protocol);
    }

    @Override
    public List<String> getCourseInventory() {
        List<String> classInv = null;
        try {
            if (!transport.isOpen()) {
                transport.open();
            }
            
            classInv = client.getCourseInventory();
        } catch (TException e) {
            e.printStackTrace(System.err);
            throw new RuntimeException(e);
        }
        
        return classInv;
    }

    @Override
    public com.ociweb.jnb.db.Course getCourse(final String courseNumber) {
        com.ociweb.jnb.db.Course classInv = null;
        try {
            if (!transport.isOpen()) {
                transport.open();
            }
            
            Course thriftCourse = client.getCourse(courseNumber);
            if (thriftCourse != null) {
                classInv = ConversionHelper.toDbCourse(thriftCourse);
            }
        } catch (TException e) {
            e.printStackTrace(System.err);
            throw new RuntimeException(e);
        } catch (CourseNotFoundException e) {
            e.printStackTrace(System.err);
            throw new RuntimeException(e);
        }
        
        return classInv;
    }

    @Override
    public void addCourse(final com.ociweb.jnb.db.Course course) {
        try {
            if (!transport.isOpen()) {
                transport.open();
            }
            
            Course thriftCourse = ConversionHelper.fromDbCourse(course);
            client.addCourse(thriftCourse);
        } catch (TException e) {
            e.printStackTrace(System.err);
            throw new RuntimeException(e);
        } catch (UnacceptableCourseException e) {
            e.printStackTrace(System.err);
            throw new RuntimeException(e);
        }
    }

    @Override
    public void deleteCourse(final String courseNumber) {
        try {
            if (!transport.isOpen()) {
                transport.open();
            }
            
            client.deleteCourse(courseNumber);
        } catch (TException e) {
            e.printStackTrace(System.err);
            throw new RuntimeException(e);
        } catch (CourseNotFoundException e) {
            e.printStackTrace(System.err);
            throw new RuntimeException(e);
        }
    }

    @Override
    public void close() {
        if (transport.isOpen()) {
            transport.close();
        }
    }
}
