package com.ociweb.jnb;

import java.awt.Graphics2D;
import java.awt.geom.Area;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.util.ArrayList;
import java.util.List;


import com.thoughtworks.xstream.XStream;
import com.thoughtworks.xstream.io.xml.XppDomDriver;

public class Assembly implements PropertyChangeListener {

	// Tags used to facilitate XStream.
	//
	private static final String XSREAM_ID = "Assembly";
	private static final String ASSEMBLY_VERSION = "assemblyVersion";
	private static final String GENERAL_AREAS = "generalAreas";
    private static final String CHANGE_SUPPORT = "changeSupport";
    private static final String XSTREAM = "xstream";

    // NON-PERSISTED data, must be regenerated on read or access.
    //
	private transient PropertyChangeSupport changeSupport = new PropertyChangeSupport(this);
    private transient XStream xstream;
	private transient Area area = null;

	// Persisted attributes.
	//
	protected List<GeneralArea> generalAreas = new ArrayList<GeneralArea>();
	protected int assemblyVersion = 1;

	public Assembly() {
		initializeXStream();
	}

	public void add(GeneralArea generalArea) {
		area = null;
		generalAreas.add(generalArea);
		generalArea.setParent(this);
		getChangeSupport().firePropertyChange(GENERAL_AREAS, null, generalArea);
	}

	public Area getArea() {
		if (area == null) {
			area = new Area();
			for (GeneralArea generalArea : generalAreas) {
				if (generalArea.isNegative()) {
					area.subtract(generalArea.getArea());
				} else {
					area.add(generalArea.getArea());
				}
			}
		}
		return area;
	}

	public void setArea(Area area) {
		this.area = area;
	}

	public void draw(Graphics2D g) {

		// fill the overall area.
		//
		g.fill(area);

		// put the outlines for each shape.
		//
		for (GeneralArea generalArea : generalAreas) {
			generalArea.draw(g);
		}
	}

	public PropertyChangeSupport getChangeSupport() {
        if (changeSupport == null) {
            changeSupport = new PropertyChangeSupport(this);
        }
        return changeSupport;
    }

	public void addPropertyChangeListener(PropertyChangeListener l) {
        getChangeSupport().addPropertyChangeListener(l);
    }

    public void removePropertyChangeListener(PropertyChangeListener l) {
        getChangeSupport().removePropertyChangeListener(l);
    }

	@Override
	public void propertyChange(PropertyChangeEvent evt) {
		area = null;
		changeSupport.firePropertyChange(evt);
	}

	private Object readResolve() {

		// set up property change listeners:
        //
        for (GeneralArea generalArea : generalAreas) {
        	generalArea.setParent(this);
        }

		return this;
	}

    private void initializeXStream() {
        if (xstream == null) {
            xstream = new XStream(new XppDomDriver());
            setupXStream(xstream);
        }
    }

    public static void setupXStream(XStream xstream) {
    	xstream.alias(XSREAM_ID, Assembly.class);
        xstream.useAttributeFor(Assembly.class, ASSEMBLY_VERSION);
        xstream.addImplicitCollection(GeneralArea.class, GENERAL_AREAS);

        xstream.omitField(GeneralArea.class, CHANGE_SUPPORT);
        xstream.omitField(GeneralArea.class, XSTREAM);

        GeneralArea.setupXStream(xstream);
    }

    public String toXMLString() {
        initializeXStream();
        return xstream.toXML(this);
    }

}
