package com.ociweb.jnb;

import java.awt.Graphics2D;
import java.awt.geom.Area;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;

import com.thoughtworks.xstream.XStream;

public abstract class GeneralArea implements PropertyChangeListener {

	// Tags used to facilitate XStream.
	//
	private static final String XTSREAM_ID_GENERAL_AREA = "GeneralArea";
	private static final String GENERAL_AREA_VERSION = "generalAreaVersion";
	private static final int GENERAL_AREA_VERSION_1 = 1;
	private static final int GENERAL_AREA_VERSION_2 = 2;
    private static final String CHANGE_SUPPORT = "changeSupport";
    private static final String GENERAL_AREA_AREA = "area";
    private static final String GERNERAL_AREA_IS_NEGATIVE = "isNegative";
    private static final String GERNERAL_AREA_ORIGIN_X = "originY";
    private static final String GERNERAL_AREA_ORIGIN_Y = "originX";

	protected static final double DEFAULT_ORIGIN_X = 0.0;
	protected static final double DEFAULT_ORIGIN_Y = 0.0;
	public static final double AREA_EPSILON = 1e-12;

    // NON-PERSISTED data, must be regenerated on read or access.
    //
	private transient PropertyChangeSupport changeSupport = new PropertyChangeSupport(this);
	private transient Area area;

	private double originX = DEFAULT_ORIGIN_X;
	private double originY = DEFAULT_ORIGIN_Y;
	private int generalAreaVersion = GENERAL_AREA_VERSION_2;
	private boolean isNegative = false;
	private Assembly parent = null;

	@SuppressWarnings("unused")
	private GeneralArea() {
	}

	protected GeneralArea(double originX, double originY) {
		setOriginX(originX);
		setOriginY(originY);
	}

	public void draw(Graphics2D g) {
		if (area != null) {
			g.draw(area);
		}
	}

	public double getOriginX() {
		return originX;
	}

	public void setOriginX(double originX) {
		if (Math.abs(this.originX - originX) > AREA_EPSILON) {
			double oldValue = this.originX;
			this.originX = originX;
			propertyChange(GERNERAL_AREA_ORIGIN_X, oldValue, originX);
		}
	}

	public double getOriginY() {
		return originY;
	}

	public void setOriginY(double originY) {
		if (Math.abs(this.originY - originY) > AREA_EPSILON) {
			double oldValue = this.originY;
			this.originY = originY;
			propertyChange(GERNERAL_AREA_ORIGIN_Y, oldValue, originY);
		}
	}

	public Area getArea() {
		if (area == null) {
			regenerate();
		}
		return area;
	}

	public void setArea(Area area) {
		this.area = area;
	}

	public boolean isNegative() {
		return isNegative;
	}

	public void setNegative(boolean isNegative) {
		if (this.isNegative != isNegative) {
			boolean oldValue = this.isNegative;
			this.isNegative = isNegative;
			propertyChange(GERNERAL_AREA_IS_NEGATIVE, oldValue, isNegative);
		}
	}

	public int getGeneralAreaVersion() {
		return generalAreaVersion;
	}

	public void setGeneralAreaVersion(int generalAreaVersion) {
		this.generalAreaVersion = generalAreaVersion;
	}

	public PropertyChangeSupport getChangeSupport() {
        if (changeSupport == null) {
            changeSupport = new PropertyChangeSupport(this);
        }
        return changeSupport;
    }

    public void setParent(Assembly newParent) {
        if (parent != newParent) {
            if (parent != null) {
                removePropertyChangeListener(parent);
            }
            parent = newParent;
            if (parent != null) {
                addPropertyChangeListener(parent);
            }
        }
    }

    public Assembly getParent() {
        return parent;
    }

    public void addPropertyChangeListener(PropertyChangeListener l) {
        getChangeSupport().addPropertyChangeListener(l);
    }

    public void removePropertyChangeListener(PropertyChangeListener l) {
        getChangeSupport().removePropertyChangeListener(l);
    }

	@Override
	public void propertyChange(PropertyChangeEvent evt) {
		if (evt.getOldValue() != null && !evt.getOldValue().equals(evt.getNewValue())) {
			area = null;
			changeSupport.firePropertyChange(evt);
		}
	}

	public void propertyChange(String name, Object oldValue, Object newValue) {
		if (!oldValue.equals(newValue)) {
			area = null;
			changeSupport.firePropertyChange(name, oldValue, newValue);
		}
	}

	private Object readResolve() {
		if (generalAreaVersion < GENERAL_AREA_VERSION_2) {
			isNegative = false;
			generalAreaVersion = GENERAL_AREA_VERSION_2;
		}
		return this;
	}

	public static void setupXStream(XStream xstream) {
		xstream.alias(XTSREAM_ID_GENERAL_AREA, GeneralArea.class);
		xstream.useAttributeFor(GENERAL_AREA_VERSION, int.class);

		xstream.omitField(GeneralArea.class, CHANGE_SUPPORT);
        xstream.omitField(GeneralArea.class, GENERAL_AREA_AREA);

        GeneralRect.setupXStream(xstream);
        GeneralCirc.setupXStream(xstream);
	}

	// abstract methods needing implementation.
	//
	protected abstract void regenerate();

}
