package test

import org.specs2.mutable._

import play.api.test._
import play.api.test.Helpers._

class ApplicationSpec extends Specification {
  import play.api.libs.json._

  "Application" should {

    "allow an authorized Ajax request" in {
      running(FakeApplication()) {
        val reqJSON = Json.obj("buttonID" -> 1)

        val Some(rsp) = route(FakeRequest(POST, "/ajaxText").withSession("user" -> "bob").withBody(reqJSON))

        status(rsp) must equalTo(OK)
        contentType(rsp) must beSome.which(_ == "application/json")
        
        val rspJSON = Json.parse(contentAsString(rsp))
        val jsText = (rspJSON \ "text").asOpt[String]
        
        jsText must not beNone
        
        jsText.get must contain("Pressed 1")
      }
    }
    
    "reject a malformed Ajax request" in {
      running(FakeApplication()) {
        val reqJSON = Json.obj("thisIsBad" -> "thisIsBad")

        val Some(rsp) = route(FakeRequest(POST, "/ajaxText").withSession("user" -> "bob").withBody(reqJSON))

        status(rsp) must equalTo(BAD_REQUEST)
        contentType(rsp) must beSome.which(_ == "application/json")
        
        val rspJSON = Json.parse(contentAsString(rsp))
        // {"status":"KO","message":{"obj.buttonID":[{"msg":"error.path.missing","args":[]}]}}
        
        val jsStatus = (rspJSON \ "status").asOpt[String]
        
        jsStatus must not beNone
        
        jsStatus.get must contain("KO")
        
        val jsError = (rspJSON \ "message" \ "obj.buttonID" \\ "msg").map(_.as[String]).headOption

        jsError must not beNone
      
        jsError.get must beEqualTo("error.path.missing")
      }
    }

    "prevent an unauthorized Ajax request" in {
      running(FakeApplication()) {
        val reqJSON = Json.obj("buttonID" -> 1)

        val Some(rsp) = route(FakeRequest(POST, "/ajaxText").withBody(reqJSON))

        status(rsp) must equalTo(FORBIDDEN)
        contentType(rsp) must beSome.which(_ == "text/html")
        contentAsString(rsp) must contain("Authorization Failed")
      }
    }
  }
}