package controllers

import play.api._
import play.api.mvc._
import play.api.data._
import play.api.data.Forms._
import be.objectify.deadbolt.scala.DeadboltActions
import security.MyDeadboltHandler

object Application extends Controller with DeadboltActions {

  val loginForm = Form(
    tuple(
      "username" -> nonEmptyText,
      "password" -> text)
      verifying ("Invalid username or password", result => result match {
        case (username, password) =>
          models.User.authenticate(username, password).isDefined
      }))

  def index = Action {
    Ok(views.html.index(loginForm))
  }

  def login = {
    Action { implicit request =>
      loginForm.bindFromRequest.fold(
        formWithErrors => { // binding failure, you retrieve the form containing errors
          Logger.error("Login failed for user " + formWithErrors.data("username"))
          BadRequest(views.html.index(formWithErrors))
        },
        user => {
          // be sure the session is updated BEFORE the view is rendered   
          val modifiedRequest = updateRequestSession(request, List(("user" -> user._1)))
          Ok(views.html.pageA(modifiedRequest)).withSession(modifiedRequest.session)
        })
    }
  }

  private def updateRequestSession(request: Request[Any], additionalSessionParams: List[(String, String)]): Request[Any] = {
    import scala.language.reflectiveCalls

    val updatedSession = additionalSessionParams.foldLeft[Session](request.session) { _ + _ }
    val existingSession = request.headers.get(SET_COOKIE).map(cookies => Session.decodeFromCookie(Cookies.decode(cookies).find(_.name == Session.COOKIE_NAME)))
    val newSession = if (existingSession.isDefined)
      existingSession.get.data.foldLeft(updatedSession) { _ + _ }
    else
      updatedSession

    val cookies = Cookies(request.headers.get(COOKIE)).cookies +
      (Session.COOKIE_NAME -> Session.encodeAsCookie(newSession))
    val headerMap = request.headers.toMap +
      (COOKIE -> Seq(Cookies.encode(cookies.values.toSeq)))
    val theHeaders = new play.api.mvc.Headers {
      val data: Seq[(String, Seq[String])] = headerMap.toSeq
    }

    Request[Any](request.copy(headers = theHeaders), request.body)
  }

  def logout =
    Action { implicit request =>
      Ok(views.html.index(loginForm)).withNewSession
    }

  def pageB = SubjectPresent(new MyDeadboltHandler) {
    Action { implicit request =>
      Ok(views.html.pageB())
    }
  }

  import play.api.libs.json._
  import play.api.libs.functional.syntax._

  val ajaxTextRds = (__ \ 'buttonID).read[Long]

  def ajaxText = SubjectPresent(new MyDeadboltHandler) {
    Action(parse.json) { request =>
      request.body.validate[Long](ajaxTextRds).map {
        case (buttonID) => {
          val text = "Pressed " + buttonID + " at " + new java.util.Date()
          Ok(Json.obj("status" -> "OK", "divID" -> buttonID, "text" -> text))
        }
      }.recoverTotal {
        e => BadRequest(Json.obj("status" -> "KO", "message" -> JsError.toFlatJson(e)))
      }
    }
  }

}