#include "DataServerConnectorLib.h"
#include <iostream>
#include <vcclr.h>
#include <sstream>
#include <exception>
#include <string>
#include "Shared/LNK4248.h"
#include <fstream>


using namespace System;
using namespace System::Threading;

// incomplete types generate LNK4248 warnings when compiled for .NET
SUPPRESS_LNK4248_CORBA
SUPPRESS_LNK4248_TAO


DataServerConnectorState::DataServerConnectorState(CORBA::ORB_ptr orb, Database_ptr database) {
	orb_ = CORBA::ORB::_duplicate(orb);
	database_ = Database::_duplicate(database);
}


DataServerConnector::DataServerConnector() : state_(NULL), thread_(nullptr), startupEvent_(false) {}

DataServerConnector::~DataServerConnector() {
	delete state_;
}

void DataServerConnector::ThreadStart(Object^ param) { 
	((DataServerConnector^)param)->Run();
}

void DataServerConnector::Start() {
	ParameterizedThreadStart^ pts = gcnew ParameterizedThreadStart(&DataServerConnector::ThreadStart);
	thread_ = gcnew Thread(pts);
	thread_->Start(this);
	// for simplicity, this blocks indefinitely on error, as the event is only signaled when 
	// startup is successful
	startupEvent_.WaitOne();
}


void DataServerConnector::Shutdown() {
	if (state_)
		state_->OrbPtr()->shutdown();
	//thread_->Join();
}


void DataServerConnector::Run() {
	int argc = 0;
	wchar_t **argv = NULL;

	try {
		// convert .NET arguments to standard argc/argv
		array<String^>^ arguments = Environment::GetCommandLineArgs();
		argc = arguments->Length;
		argv = new wchar_t *[argc];
		for (int i=0; i<argc; i++) {
			pin_ptr<const wchar_t> arg = PtrToStringChars(arguments[i]);
			argv[i] = _wcsdup(arg);
		}

		// initialize the ORB
		CORBA::ORB_var orb = CORBA::ORB_init(argc, argv);

		// USE ORBINITREF - start server with -ORBListenEndpoints iiop://machine:port
		// and client with -ORBInitRef DataServer=corbaloc:iiop:machine:port/DataServer
		CORBA::Object_var database_obj = orb->resolve_initial_references("DataServer");
		if (CORBA::is_nil(database_obj.in())) 
			throw std::exception("Could not get the Database IOR");

		// narrow the IOR to a Database object reference.
		Database_var database = Database::_narrow(database_obj.in());
		if (CORBA::is_nil(database.in()))
			throw std::exception("IOR was not a Database object reference");

		// save the references via a pointer to an unmanaged class
		state_ = new DataServerConnectorState(orb, database);

		// good to go - tell the outside world
		startupEvent_.Set();

		// run the ORB - need to wait anyway for DDS, so may as well process AMI calls, etc.
		orb->run();
		orb->destroy();
	}
	catch (CORBA::Exception& ex) {
		std::stringstream ss;
		ss << "Exception: " << ex;
		throw gcnew DataConnectorException(gcnew String(ss.str().c_str()));
	}
	catch (std::exception& ex) {
		std::stringstream ss;
		ss << "Exception: " << ex.what();
		throw gcnew DataConnectorException(gcnew String(ss.str().c_str()));
	}
}


// *** .NET wrappers for CORBA methods ***
// DataConnector must have a pure .NET interface, so arguments must be CLR types

bool DataServerConnector::CreateItem(String ^description, Int64 %id) {
	try {
		pin_ptr<const wchar_t> cppDescription = PtrToStringChars(description);
		CORBA::WString_var desc = CORBA::wstring_dup(cppDescription);
		CORBA::LongLong cid;
		CORBA::Boolean result = state_->DatabasePtr()->CreateItem(desc, cid);
		id = cid;
		return result;
	} catch (CORBA::Exception &ex) {
		std::stringstream ss;
		ss << "Exception: " << ex;
		throw gcnew DataConnectorException(gcnew String(ss.str().c_str()));
	}
}

bool DataServerConnector::ReadItem(Int64 id, String^% description) {
	try {
		CORBA::WString_var desc;
		CORBA::Boolean result = state_->DatabasePtr()->ReadItem(id, desc);
		if (result)
			description = gcnew String(desc.in());
		return result;
	} catch (CORBA::Exception &ex) {
		std::stringstream ss;
		ss << "Exception: " << ex;
		throw gcnew DataConnectorException(gcnew String(ss.str().c_str()));
	}
}

bool DataServerConnector::UpdateItem(Int64 id, String^ description) {
	try {
		pin_ptr<const wchar_t> cppDescription = PtrToStringChars(description);
		CORBA::WString_var desc = CORBA::wstring_dup(cppDescription);
		return state_->DatabasePtr()->UpdateItem(id, desc);
	} catch (CORBA::Exception &ex) {
		std::stringstream ss;
		ss << "Exception: " << ex;
		throw gcnew DataConnectorException(gcnew String(ss.str().c_str()));
	}
}

bool DataServerConnector::DeleteItem(Int64 id) {
	try {
		return state_->DatabasePtr()->DeleteItem(id);
	} catch (CORBA::Exception &ex) {
		std::stringstream ss;
		ss << "Exception: " << ex;
		throw gcnew DataConnectorException(gcnew String(ss.str().c_str()));
	}
}
