﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Reflection;
using System.IO;

namespace DDSGenerator
{
    public class DDSGen
    {
        const string NO_NAMESPACE = "NO_NAMESPACE";

        public DDSGen() { }

        Dictionary<string, List<Type>> GetDDSTypes(Assembly a)
        {
            Dictionary<string, List<Type>> ddsTypes = new Dictionary<string, List<Type>>();

            // read valuetypes, sort by namespace
            foreach (Type type in a.GetTypes())
            {
                if (type.IsValueType)
                {
                    foreach (Attribute attr in type.GetCustomAttributes(true))
                    {
                        DCPSDataTypeAttribute dataAttr = attr as DCPSDataTypeAttribute;
                        if (null != dataAttr)
                        {
                            // NO_NAMESPACE as can't use a null as a key
                            string ns = String.IsNullOrEmpty(type.Namespace) ? NO_NAMESPACE : type.Namespace;
                            if (!ddsTypes.ContainsKey(ns))
                                ddsTypes.Add(ns, new List<Type>());
                            ddsTypes[ns].Add(type);
                        }
                    }
                }
            }

            return ddsTypes;
        }


        void GenerateIDLFile(Dictionary<string, List<Type>> ddsTypes, string baseName, string outputDirectory)
        {
            if (ddsTypes.Count == 0)
                return;  // nothing to do

            TextWriter idlFile = new StreamWriter(outputDirectory + "\\" + baseName + ".idl");
            foreach (string ns in ddsTypes.Keys)
            {
                if (ns != NO_NAMESPACE)
                    idlFile.WriteLine("module " + ns + " {");

                foreach (Type ddsType in ddsTypes[ns])
                {
                    string fullTypeName = "";
                    if (ns != NO_NAMESPACE)
                        fullTypeName = ns + "::";
                    fullTypeName += ddsType.Name;

                    idlFile.WriteLine("#pragma DCPS_DATA_TYPE \"" + fullTypeName + "\"");


                    // build key list
                    List<FieldInfo> keys = new List<FieldInfo>();
                    foreach (FieldInfo fi in ddsType.GetFields())
                    {
                        foreach (Attribute attr2 in fi.GetCustomAttributes(true))
                        {
                            DCPSKeyAttribute keyAttr = attr2 as DCPSKeyAttribute;
                            if (null != keyAttr)
                                keys.Add(fi);
                        }
                    }
                    foreach (FieldInfo fi in keys)
                        idlFile.WriteLine("#pragma DCPS_DATA_KEY \"" + fullTypeName + " " + fi.Name + "\"");

                    // structure
                    idlFile.WriteLine("struct " + ddsType.Name + " {");
                    foreach (FieldInfo fi in ddsType.GetFields())
                    {
                        idlFile.Write("  ");
                        if (fi.FieldType == typeof(String))
                            idlFile.Write("string");
                        else if (fi.FieldType == typeof(int))
                            idlFile.Write("long");
                        else if (fi.FieldType == typeof(long))
                            idlFile.Write("long long");
                        else
                            Console.WriteLine(fi.Name + " with a type of " + fi.FieldType + " is not supported");
                        idlFile.WriteLine(" " + fi.Name + ";");
                    }
                    idlFile.WriteLine("};");
                    idlFile.WriteLine();
                }

                if (ns != NO_NAMESPACE)
                    idlFile.WriteLine("};");
            }
            idlFile.Close();
        }


        void WriteTypeSupportImplIncludes(TextWriter ddsImplHFile, Dictionary<string, List<Type>> ddsTypes)
        {
            // #include "MessengerTypeSupportImpl.h"
            foreach (string ns in ddsTypes.Keys)
                if (ns == NO_NAMESPACE)
                    ddsImplHFile.WriteLine("#include \"" + "" + "TypeSupportImpl.h\"");  // check this
                else
                    ddsImplHFile.WriteLine("#include \"" + ns + "TypeSupportImpl.h\"");  
        }

        void WriteTypeTraits(TextWriter ddsImplHFile, Dictionary<string, List<Type>> ddsTypes)
        {
            ddsImplHFile.WriteLine("#include \"Common/TypeTraits.h\"");
            foreach (string ns in ddsTypes.Keys)
            {
                foreach (Type ddsType in ddsTypes[ns])
                {
                    string fullTypeName, fullTypeNameSep;
                    GetFullTypeName(ns, ddsType.Name, out fullTypeName, out fullTypeNameSep);
                    ddsImplHFile.WriteLine("DEFINE_TYPETRAITS(" + fullTypeNameSep + ")");
                }
            }
        }

        void WriteNetTypes(TextWriter ddsImplHFile, TextWriter ddsImplCPPFile, Dictionary<string, List<Type>> ddsTypes)
        {
            /*
                // DDSImpl.h
                namespace MessengerNet {
	                public value class MessageNet
	                {
	                public:
		                System::String^ from;
		                System::String^ subject;
		                System::Int32 subject_id;
		                System::String^ text;
		                System::Int32 count;
	                };
                }
            */

            foreach (string ns in ddsTypes.Keys)
            {
                if (ns != NO_NAMESPACE)
                    ddsImplHFile.WriteLine("namespace " + ns + "Net {");

                foreach (Type ddsType in ddsTypes[ns])
                {
                    ddsImplHFile.WriteLine("    public value class " + ddsType.Name + "Net {");
                    ddsImplHFile.WriteLine("    public:");

                    foreach (FieldInfo fi in ddsType.GetFields())
                    {
                        ddsImplHFile.Write("        ");
                        if (fi.FieldType == typeof(String))
                            ddsImplHFile.Write("System::String^");
                        else if (fi.FieldType == typeof(int))
                            ddsImplHFile.Write("System::Int32");
                        else if (fi.FieldType == typeof(long))
                            ddsImplHFile.Write("System::Int64");
                        ddsImplHFile.WriteLine(" " + fi.Name + ";");
                    }

                    ddsImplHFile.WriteLine("    };");
                }

                if (ns != NO_NAMESPACE)
                    ddsImplHFile.WriteLine("}");
            }

            ddsImplCPPFile.WriteLine("#include \"DDSImpl.h\"");

            foreach (string ns in ddsTypes.Keys)
            {
                foreach (Type ddsType in ddsTypes[ns])
                {
                    string fullTypeName, fullTypeNameSep, fullNetTypeName, fullNetTypeNameSep;
                    GetFullTypeName(ns, ddsType.Name, out fullTypeName, out fullTypeNameSep);
                    GetFullNetTypeName(ns, ddsType.Name, out fullNetTypeName, out fullNetTypeNameSep);

                    /*
                        // DDSImpl.h
                        Messenger::Message Convert(const gcroot<MessengerNet::MessageNet> &messageNetParam);
                        gcroot<MessengerNet::MessageNet> Convert(const Messenger::Message &message);\
                    */

                    ddsImplHFile.WriteLine(fullTypeNameSep + " Convert(const gcroot<" + fullNetTypeNameSep + "> &sampleNetParam);");
                    ddsImplHFile.WriteLine("gcroot<" + fullNetTypeNameSep + "> Convert(const " + fullTypeNameSep + " &sample);");

                    /*
                        // DDSImpl.cpp                    
                        Messenger::Message Convert(const gcroot<MessengerNet::MessageNet> &sampleNetParam) {
	                        MessengerNet::MessageNet sampleNet = (MessengerNet::MessageNet)sampleNetParam;
	                        Messenger::Message sample;
	                        sample.subject_id = sampleNet.subject_id;
	                        sample.from	= Convert(sampleNet.from);
	                        sample.subject = Convert(sampleNet.subject);
	                        sample.text = Convert(sampleNet.text);
	                        sample.count = sampleNet.count;
	                        return sample;
                        }
                     */

                    ddsImplCPPFile.WriteLine(fullTypeNameSep + " Convert(const gcroot<" + fullNetTypeNameSep + "> &sampleNetParam) {");
                    ddsImplCPPFile.WriteLine("    " + fullNetTypeNameSep + " sampleNet = (" + fullNetTypeNameSep + ")sampleNetParam;");
                    ddsImplCPPFile.WriteLine("    " + fullTypeNameSep + " sample;");

                    foreach (FieldInfo fi in ddsType.GetFields())
                    {
                        ddsImplCPPFile.Write("    sample." + fi.Name + " = ");
                        if (fi.FieldType == typeof(String))
                            ddsImplCPPFile.WriteLine("Convert(sampleNet." + fi.Name + ");");
                        else
                            ddsImplCPPFile.WriteLine("sampleNet." + fi.Name + ";");
                    }

                    ddsImplCPPFile.WriteLine("    return sample;");
                    ddsImplCPPFile.WriteLine("}");

                    ddsImplCPPFile.WriteLine();


                    /*
                       // DDSImpl.cpp                    
                       gcroot<MessengerNet::MessageNet> Convert(const Messenger::Message &sample) {
	                        MessengerNet::MessageNet sampleNet;
	                        sampleNet.subject_id = sample.subject_id;
	                        sampleNet.from = Convert(sample.from);
                            sampleNet.subject = Convert(sample.subject);
                            sampleNet.text = Convert(sample.text);
                            sampleNet.count = sample.count;
	                        return sampleNet;
                        }
                    */
                    ddsImplCPPFile.WriteLine("gcroot<" + fullNetTypeNameSep + "> Convert(const " + fullTypeNameSep + " &sample) {");
                    ddsImplCPPFile.WriteLine("    " + fullNetTypeNameSep + " sampleNet;");

                    foreach (FieldInfo fi in ddsType.GetFields())
                    {
                        ddsImplCPPFile.Write("    sampleNet." + fi.Name + " = ");
                        if (fi.FieldType == typeof(String))
                            ddsImplCPPFile.WriteLine("Convert(sample." + fi.Name + ");");
                        else
                            ddsImplCPPFile.WriteLine("sample." + fi.Name + ";");
                    }
                  
                    ddsImplCPPFile.WriteLine("    return sampleNet;");
                    ddsImplCPPFile.WriteLine("}");

                    ddsImplCPPFile.WriteLine();
                }
            }
        }

        void GetFullTypeName(string ns, string name, out string fullTypeName, out string fullTypeNameSep)
        {
            fullTypeName = "";
            fullTypeNameSep = "";
            if (ns != NO_NAMESPACE)
            {
                fullTypeName = ns;
                fullTypeNameSep = ns + "::";
            }
            fullTypeName += name;
            fullTypeNameSep += name;
        }

        void GetFullNetTypeName(string ns, string name, out string fullNetTypeName, out string fullNetTypeNameSep)
        {
            fullNetTypeName = "";
            fullNetTypeNameSep = "";
            if (ns != NO_NAMESPACE)
            {
                fullNetTypeName = ns + "Net";
                fullNetTypeNameSep = ns + "Net::";
            }
            fullNetTypeName += name + "Net";
            fullNetTypeNameSep += name + "Net";
        }

        void GenerateDataReaderListener(TextWriter ddsImplHFile, Dictionary<string, List<Type>> ddsTypes)
        {
            /*
                class MessengerMessageDataReaderListenerImpl : public DataReaderListenerImplBase<Messenger::MessageDataReader, Messenger::Message> {
	                gcroot<EventManager<MessengerNet::MessageNet>^> eventManager_;
                public:
	                MessengerMessageDataReaderListenerImpl(gcroot<EventManager<MessengerNet::MessageNet>^> eventManager) : eventManager_(eventManager) {}

	                void Process(const Messenger::Message &sample) {
		                eventManager_->Process(eventManager_, 
			                gcnew ProcessEventArgs<MessengerNet::MessageNet>(Convert(sample)));
	                }

	                void AddHandler(gcroot<EventManager<MessengerNet::MessageNet>::ProcessEventHandler^> handler) {
		                eventManager_->Process += handler;
	                }
                };
            */

            foreach (string ns in ddsTypes.Keys)
            {
                foreach (Type ddsType in ddsTypes[ns])
                {
                    string fullTypeName, fullTypeNameSep, fullNetTypeName, fullNetTypeNameSep;
                    GetFullTypeName(ns, ddsType.Name, out fullTypeName, out fullTypeNameSep);
                    GetFullNetTypeName(ns, ddsType.Name, out fullNetTypeName, out fullNetTypeNameSep);

                    ddsImplHFile.WriteLine("class " + fullTypeName + "DataReaderListenerImpl : public DataReaderListenerImplBase<" + fullTypeNameSep + "DataReader, " + fullTypeNameSep + "> {");
                    ddsImplHFile.WriteLine("    gcroot<EventManager<" + fullNetTypeNameSep + ">^> eventManager_;");
                    ddsImplHFile.WriteLine("public:");
                    ddsImplHFile.WriteLine("    " + fullTypeName + "DataReaderListenerImpl(gcroot<EventManager<" + fullNetTypeNameSep + ">^> eventManager) : eventManager_(eventManager) {}");
                    ddsImplHFile.WriteLine("    void Process(const " + fullTypeNameSep + " &sample) {");
                    ddsImplHFile.WriteLine("        eventManager_->Process(eventManager_,");
                    ddsImplHFile.WriteLine("            gcnew ProcessEventArgs<" + fullNetTypeNameSep + ">(Convert(sample)));");
                    ddsImplHFile.WriteLine("    }");
                    ddsImplHFile.WriteLine("    void AddHandler(gcroot<EventManager<" + fullNetTypeNameSep + ">::ProcessEventHandler^> handler) {");
                    ddsImplHFile.WriteLine("        eventManager_->Process += handler;");
                    ddsImplHFile.WriteLine("    }");
                    ddsImplHFile.WriteLine("};");
                    ddsImplHFile.WriteLine();
                }
            }
        }

        void GenerateReaderWriterMaps(TextWriter ddsImplHFile, Dictionary<string, List<Type>> ddsTypes)
        {
            foreach (string ns in ddsTypes.Keys)
            {
                foreach (Type ddsType in ddsTypes[ns])
                {
                    string fullTypeName, fullTypeNameSep, fullNetTypeName, fullNetTypeNameSep;
                    GetFullTypeName(ns, ddsType.Name, out fullTypeName, out fullTypeNameSep);
                    GetFullNetTypeName(ns, ddsType.Name, out fullNetTypeName, out fullNetTypeNameSep);

                    /*
                        typedef Writer<Messenger::Message> MessengerMessageDataWriter;
                        class MessengerMessageDataWriterMap : public EntityMap<MessengerMessageDataWriter> {
	                        virtual MessengerMessageDataWriter Create(DDSBase *ddsBase, ::DDS::DomainId_t domainId, const char *topic_name) {
		                        return ddsBase->CreateWriter<Messenger::Message>(domainId, topic_name);
	                        }
                        };
                    */
                    ddsImplHFile.WriteLine("typedef Writer<" + fullTypeNameSep + "> " + fullTypeName + "DataWriter;");
                    ddsImplHFile.WriteLine("class " + fullTypeName + "DataWriterMap : public EntityMap<" + fullTypeName + "DataWriter> {");
                    ddsImplHFile.WriteLine("    virtual " + fullTypeName + "DataWriter Create(DDSBase *ddsBase, ::DDS::DomainId_t domainId, const char *topic_name) {");
                    ddsImplHFile.WriteLine("        return ddsBase->CreateWriter<" + fullTypeNameSep + ">(domainId, topic_name);");
                    ddsImplHFile.WriteLine("    }");
                    ddsImplHFile.WriteLine("};");


                    /*
                        typedef Reader<Messenger::Message, MessengerMessageDataReaderListenerImpl> MessengerMessageDataReader;
                        class MessengerMessageDataReaderMap : public EntityMap<MessengerMessageDataReader> {
	                        virtual MessengerMessageDataReader Create(DDSBase *ddsBase, ::DDS::DomainId_t domainId, const char *topic_name) {
		                        return ddsBase->CreateReader<Messenger::Message>(domainId, topic_name, new MessengerMessageDataReaderListenerImpl(gcnew EventManager<MessengerNet::MessageNet>()));
	                        }
                        };
                    */
                    ddsImplHFile.WriteLine("typedef Reader<" + fullTypeNameSep + ", " + fullTypeName + "DataReaderListenerImpl> " + fullTypeName + "DataReader;");
                    ddsImplHFile.WriteLine("class " + fullTypeName + "DataReaderMap : public EntityMap<" + fullTypeName + "DataReader> {");
                    ddsImplHFile.WriteLine("    virtual " + fullTypeName + "DataReader Create(DDSBase *ddsBase, ::DDS::DomainId_t domainId, const char *topic_name) {");
                    ddsImplHFile.WriteLine("        return ddsBase->CreateReader<" + fullTypeNameSep + ">(domainId, topic_name, new " + fullTypeName + "DataReaderListenerImpl(gcnew EventManager<" + fullNetTypeNameSep + ">()));");
                    ddsImplHFile.WriteLine("    }");
                    ddsImplHFile.WriteLine("};");

                    ddsImplHFile.WriteLine();
                }
            }
        }



        void GenerateDDSImplClass(TextWriter ddsImplHFile, Dictionary<string, List<Type>> ddsTypes)
        {
            /*
                class DDSImpl : public DDSBase {
                    MessengerMessageDataWriterMap mapMessengerMessageDataWriter_;
                    MessengerMessageDataReaderMap mapMessengerMessageDataReader_;

                public:
                    DDSImpl(int argc, ACE_TCHAR *argv[]) : DDSBase(argc, argv) {}

                    DDS::ReturnCode_t Publish(::DDS::DomainId_t domainId, const char *topic_name, gcroot<MessengerNet::MessageNet> sample) {
                        return mapMessengerMessageDataWriter_.Get(this, domainId, topic_name).Write(Convert(sample));
                    }

                    bool MessengerMessageWaitForSubscriber(::DDS::DomainId_t domainId, const char *topic_name) { 
                        return mapMessengerMessageDataWriter_.Get(this, domainId, topic_name).WaitForSubscriber(); 
                    }
                    bool MessengerMessageWaitForAcknowledgements(::DDS::DomainId_t domainId, const char *topic_name) { 
                        return mapMessengerMessageDataWriter_.Get(this, domainId, topic_name).WaitForAcknowledgements(); 
                    }

                    void Subscribe(::DDS::DomainId_t domainId, const char *topic_name, gcroot<EventManager<MessengerNet::MessageNet>::ProcessEventHandler^> handler) {
                        mapMessengerMessageDataReader_.Get(this, domainId, topic_name).GetListener()->AddHandler(handler);
                    }

                    bool MessengerMessageWaitForPublisherToComplete(::DDS::DomainId_t domainId, const char *topic_name) {
                        return mapMessengerMessageDataReader_.Get(this, domainId, topic_name).WaitForPublisherToComplete();
                    }
                };
                */

            ddsImplHFile.WriteLine("class DDSImpl : public DDSBase {");
            foreach (string ns in ddsTypes.Keys)
            {
                foreach (Type ddsType in ddsTypes[ns])
                {
                    string fullTypeName, fullTypeNameSep, fullNetTypeName, fullNetTypeNameSep;
                    GetFullTypeName(ns, ddsType.Name, out fullTypeName, out fullTypeNameSep);
                    GetFullNetTypeName(ns, ddsType.Name, out fullNetTypeName, out fullNetTypeNameSep);

                    ddsImplHFile.WriteLine("    " + fullTypeName + "DataWriterMap map" + fullTypeName + "DataWriter_;");
                    ddsImplHFile.WriteLine("    " + fullTypeName + "DataReaderMap map" + fullTypeName + "DataReader_;");
                }
            }

            ddsImplHFile.WriteLine("public:");
            ddsImplHFile.WriteLine("    DDSImpl(int argc, ACE_TCHAR *argv[]) : DDSBase(argc, argv) {}");

            foreach (string ns in ddsTypes.Keys)
            {
                foreach (Type ddsType in ddsTypes[ns])
                {
                    string fullTypeName, fullTypeNameSep, fullNetTypeName, fullNetTypeNameSep;
                    GetFullTypeName(ns, ddsType.Name, out fullTypeName, out fullTypeNameSep);
                    GetFullNetTypeName(ns, ddsType.Name, out fullNetTypeName, out fullNetTypeNameSep);

                    ddsImplHFile.WriteLine("    DDS::ReturnCode_t Publish(::DDS::DomainId_t domainId, const char *topic_name, gcroot<" + fullNetTypeNameSep + "> sample) {");
                    ddsImplHFile.WriteLine("        return map" + fullTypeName + "DataWriter_.Get(this, domainId, topic_name).Write(Convert(sample));");
                    ddsImplHFile.WriteLine("    }");

                    ddsImplHFile.WriteLine("    bool " + fullTypeName + "WaitForSubscriber(::DDS::DomainId_t domainId, const char *topic_name) {");
                    ddsImplHFile.WriteLine("        return map" + fullTypeName + "DataWriter_.Get(this, domainId, topic_name).WaitForSubscriber();");
                    ddsImplHFile.WriteLine("    }");

                    ddsImplHFile.WriteLine("    bool " + fullTypeName + "WaitForAcknowledgements(::DDS::DomainId_t domainId, const char *topic_name) {");
                    ddsImplHFile.WriteLine("        return map" + fullTypeName + "DataWriter_.Get(this, domainId, topic_name).WaitForAcknowledgements();");
                    ddsImplHFile.WriteLine("    }");

                    ddsImplHFile.WriteLine("    void Subscribe(::DDS::DomainId_t domainId, const char *topic_name, gcroot<EventManager<" + fullNetTypeNameSep + ">::ProcessEventHandler^> handler) {");
                    ddsImplHFile.WriteLine("        map" + fullTypeName + "DataReader_.Get(this, domainId, topic_name).GetListener()->AddHandler(handler);");
                    ddsImplHFile.WriteLine("    }");

                    ddsImplHFile.WriteLine("    bool " + fullTypeName + "WaitForPublisherToComplete(::DDS::DomainId_t domainId, const char *topic_name) {");
                    ddsImplHFile.WriteLine("        return map" + fullTypeName + "DataReader_.Get(this, domainId, topic_name).WaitForPublisherToComplete();");
                    ddsImplHFile.WriteLine("    }");
                }
            }

            ddsImplHFile.WriteLine("};");
        }

        void GenerateDDSNetClass(TextWriter ddsImplHFile, Dictionary<string, List<Type>> ddsTypes)
        {

            /*
            public ref class DDSNet {
                DDSImpl *pDDSImpl;

            public:
                DDSNet() {
                    // convert .NET arguments to standard argc/argv
                    int argc;
                    char **argv;
                    GetArguments(argc, argv);
                    pDDSImpl = new DDSImpl(argc, argv);
                }
                ~DDSNet() { 
                    delete pDDSImpl;
                }

                int Publish(int domainId, System::String^ topic_name, MessengerNet::MessageNet sample) {
                    return pDDSImpl->Publish(domainId, ConvertToString(topic_name).c_str(), sample);
                }

                bool MessengerMessageWaitForSubscriber(int domainId, System::String^ topic_name) {
                    return pDDSImpl->MessengerMessageWaitForSubscriber(domainId, ConvertToString(topic_name).c_str());
                }

                bool MessengerMessageWaitForAcknowledgements(int domainId, System::String^ topic_name) { 
                    return pDDSImpl->MessengerMessageWaitForAcknowledgements(domainId, ConvertToString(topic_name).c_str());
                }

                void Subscribe(int domainId, System::String^ topic_name, EventManager<MessengerNet::MessageNet>::ProcessEventHandler^ handler) {
                    pDDSImpl->Subscribe(domainId, ConvertToString(topic_name).c_str(), handler);
                }

                bool MessengerMessageWaitForPublisherToComplete(int domainId, System::String^ topic_name) {
                    return pDDSImpl->MessengerMessageWaitForPublisherToComplete(domainId, ConvertToString(topic_name).c_str());
                }
            };
            */

            ddsImplHFile.WriteLine("public ref class DDSNet {");
            ddsImplHFile.WriteLine("    DDSImpl *pDDSImpl;");
            ddsImplHFile.WriteLine("public:");
            ddsImplHFile.WriteLine("    DDSNet() {");
            ddsImplHFile.WriteLine("        int argc;");
            ddsImplHFile.WriteLine("        char **argv;");
            ddsImplHFile.WriteLine("        GetArguments(argc, argv);");
            ddsImplHFile.WriteLine("        pDDSImpl = new DDSImpl(argc, argv);");
            ddsImplHFile.WriteLine("    }");
            ddsImplHFile.WriteLine("    ~DDSNet() {");
            ddsImplHFile.WriteLine("        delete pDDSImpl;");
            ddsImplHFile.WriteLine("    }");

            foreach (string ns in ddsTypes.Keys)
            {
                foreach (Type ddsType in ddsTypes[ns])
                {
                    string fullTypeName, fullTypeNameSep, fullNetTypeName, fullNetTypeNameSep;
                    GetFullTypeName(ns, ddsType.Name, out fullTypeName, out fullTypeNameSep);
                    GetFullNetTypeName(ns, ddsType.Name, out fullNetTypeName, out fullNetTypeNameSep);

                    ddsImplHFile.WriteLine("    int Publish(::DDS::DomainId_t domainId, System::String^ topic_name, " + fullNetTypeNameSep + " sample) {");
                    ddsImplHFile.WriteLine("        NET_RETHROW_EXCEPTION(return pDDSImpl->Publish(domainId, ConvertToString(topic_name).c_str(), sample));");
                    ddsImplHFile.WriteLine("    }");

                    ddsImplHFile.WriteLine("    bool " + fullTypeName + "WaitForSubscriber(int domainId, System::String^ topic_name) {");
                    ddsImplHFile.WriteLine("        NET_RETHROW_EXCEPTION(return pDDSImpl->" + fullTypeName + "WaitForSubscriber(domainId, ConvertToString(topic_name).c_str()));");
                    ddsImplHFile.WriteLine("    }");

                    ddsImplHFile.WriteLine("    bool " + fullTypeName + "WaitForAcknowledgements(int domainId, System::String^ topic_name) {");
                    ddsImplHFile.WriteLine("        NET_RETHROW_EXCEPTION(return pDDSImpl->" + fullTypeName + "WaitForAcknowledgements(domainId, ConvertToString(topic_name).c_str()));");
                    ddsImplHFile.WriteLine("    }");
           
                    ddsImplHFile.WriteLine("    void Subscribe(int domainId, System::String^ topic_name, EventManager<" + fullNetTypeNameSep + ">::ProcessEventHandler^ handler) {");
                    ddsImplHFile.WriteLine("        NET_RETHROW_EXCEPTION(pDDSImpl->Subscribe(domainId, ConvertToString(topic_name).c_str(), handler));");
                    ddsImplHFile.WriteLine("    }");

                    ddsImplHFile.WriteLine("    bool " + fullTypeName + "WaitForPublisherToComplete(int domainId, System::String^ topic_name) {");
                    ddsImplHFile.WriteLine("        NET_RETHROW_EXCEPTION(return pDDSImpl->" + fullTypeName + "WaitForPublisherToComplete(domainId, ConvertToString(topic_name).c_str()));");
                    ddsImplHFile.WriteLine("    }");
                }
            }

            ddsImplHFile.WriteLine("};");


        }




        void GenerateDDSImpl(Dictionary<string, List<Type>> ddsTypes, string outputDirectory)
        {
            TextWriter ddsImplHFile = new StreamWriter(outputDirectory + "\\DDSImpl.h");
            TextWriter ddsImplCPPFile = new StreamWriter(outputDirectory + "\\DDSImpl.cpp");

            ddsImplHFile.WriteLine("#ifndef __DDSIMPL_H__");
            ddsImplHFile.WriteLine("#define __DDSIMPL_H__");
            ddsImplHFile.WriteLine();
            WriteTypeSupportImplIncludes(ddsImplHFile, ddsTypes);
            WriteTypeTraits(ddsImplHFile, ddsTypes);
            ddsImplHFile.WriteLine("#include \"Common/Common.h\"");
            ddsImplHFile.WriteLine();
            WriteNetTypes(ddsImplHFile, ddsImplCPPFile, ddsTypes);
            GenerateDataReaderListener(ddsImplHFile, ddsTypes);
            GenerateReaderWriterMaps(ddsImplHFile, ddsTypes);
            ddsImplHFile.WriteLine();
            GenerateDDSImplClass(ddsImplHFile, ddsTypes);
            GenerateDDSNetClass(ddsImplHFile, ddsTypes);
            ddsImplHFile.WriteLine("#endif");
            ddsImplHFile.Close();
            ddsImplCPPFile.Close();
        }



        public void Generate(string structAssembly, string baseName, string outputDirectory)
        {
            Dictionary<string, List<Type>> ddsTypes = GetDDSTypes(Assembly.LoadFrom(structAssembly));
            GenerateIDLFile(ddsTypes, baseName, outputDirectory);
            GenerateDDSImpl(ddsTypes, outputDirectory);
        }

        static bool GetOption(string opt, out string val, ref List<string> argList)
        {
            val = "";
            for (int i=0; i<argList.Count; i++)
                if (argList[i] == opt)
                {
                    if (i + 1 < argList.Count - 1)
                    {
                        val = argList[i + 1];
                        argList.RemoveAt(i + 1);
                        argList.RemoveAt(i);
                        return true;
                    }
                    else
                        Console.WriteLine("Option " + opt + " found, but it has no value");
                }
            return false;
        }

        static class Program
        {
            [STAThread]
            static void Main(string[] args)
            {
                try
                {
                    if (args.Length == 0)
                    {
                        Console.WriteLine("Usage: DDSGen -o basename -d output_dir assembly.dll");
                        return;
                    }

                    List<string> argList = new List<string>();
                    foreach (string arg in args)
                        argList.Add(arg);

                    string baseName;
                    if (!GetOption("-o", out baseName, ref argList))
                    {
                        Console.WriteLine("A base name for the IDL output file (-o) must be specified.");
                        return;
                    }
                    string outputDirectory;
                    if (!GetOption("-d", out outputDirectory, ref argList))
                    {
                        Console.WriteLine("An output directory (-d) must be specified.");
                        return;
                    }

                    if (argList.Count != 1)
                    {
                        Console.WriteLine("An assembly containing annotated structures must be provided.");
                        return;
                    }

                    DDSGen g = new DDSGen();
                    g.Generate(argList[0], baseName, outputDirectory);
                }
                catch (Exception e)
                {
                    Console.WriteLine(e);
                    throw;
                }
            }
        }
    }
}

/*

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Reflection;
using System.Diagnostics;

namespace attrib
{

    [AttributeUsage(AttributeTargets.Struct, AllowMultiple = false, Inherited = false) ]
    public class DDSAttribute : Attribute
    {
        public DDSAttribute(String Description_in)
        {
            this.description = Description_in;
        }
        protected String description;
        public String Description
        {
            get 
            {
                return this.description;
            }            
        }

        protected int anotherProperty = 0;
        public int AnotherProperty
        {
            get { return anotherProperty; }
            set { anotherProperty = value; }
        }
    }

    [AttributeUsage(AttributeTargets.Field, AllowMultiple = false, Inherited = false)]
    public class KeyAttribute : Attribute
    {
        public KeyAttribute(String Description_in)
        {
            this.description = Description_in;
        }
        protected String description;
        public String Description
        {
            get
            {
                return this.description;
            }
        }

        protected int anotherProperty = 0;
        public int AnotherProperty
        {
            get { return anotherProperty; }
            set { anotherProperty = value; }
        }
    }


    [DDS("test1")]
    public struct Data1
    {
        [Key("the_key")]
        public int x1;
        public double y1;
    }

    [DDS("test2", AnotherProperty=3)]
    public struct Data2
    {
        public int x2;
        public double y2;
    }

    class Program
    {
        static void Main(string[] args)
        {

            // Assembly a = Assembly.GetExecutingAssembly();
            String assemblyName;
            Process p = Process.GetCurrentProcess();
            assemblyName = p.ProcessName + ".exe";
            Assembly a = Assembly.LoadFrom(assemblyName);

            foreach (Attribute attr in a.GetCustomAttributes(true))
            {
                DDSAttribute DDSAttr = attr as DDSAttribute;
                if (null != DDSAttr)
                {
                    Console.WriteLine("Description of {0}:\n{1} {2}",
                                      assemblyName, DDSAttr.Description, DDSAttr.AnotherProperty);
                }
            }

            foreach (Type t in a.GetTypes())
            {
                if (t.IsValueType)
                {
                    foreach (Attribute attr in t.GetCustomAttributes(true))
                    {
                        DDSAttribute DDSAttr = attr as DDSAttribute;
                        if (null != DDSAttr)
                        {
                            Console.WriteLine("Description of {0}:\nDDSAttr {1} {2}",
                                              assemblyName, DDSAttr.Description, DDSAttr.AnotherProperty);
                        }
                    }

                    foreach (MemberInfo mi in t.GetMembers())
                    {
                        foreach (Attribute attr in mi.GetCustomAttributes(true))
                        {
                            KeyAttribute KeyAttr = attr as KeyAttribute;
                            if (null != KeyAttr)
                            {
                                Console.WriteLine("    KeyAttr {1} {2}",
                                                  assemblyName, KeyAttr.Description, KeyAttr.AnotherProperty);
                            }
                        }

                    }
                }
            }

        }
    }
}
*/