#include <iostream>
#include <fstream>
#include <string>
#include <tchar.h>
#include <tao/IORTable/IORTable.h>
#include "Database_i.h"
#include "Shared/LNK4248.h"

// for DDS
#include "DatabaseNotification/DatabaseNotificationTypeSupportImpl.h"
#include <dds/DCPS/Service_Participant.h>
#include <dds/DCPS/Marked_Default_Qos.h>
#include <dds/DCPS/PublisherImpl.h>
#include <dds/DCPS/transport/framework/TheTransportFactory.h>

// for DCPSInfoRepo colocation
#include "InfoRepoColoc.h"

// for argument creation
#include <ace/Arg_Shifter.h>

using namespace System;

// incomplete types generate LNK4248 warnings when compiled for .NET
SUPPRESS_LNK4248_CORBA

// to allow the server to be shut down gracefully with ctrl-c
CORBA::ORB_ptr gOrb_ = 0;
static BOOL WINAPI ControlCHandler(DWORD /*type*/) {
	std::cout << "Quitting because of ctrl-c" << std::endl;
	if (gOrb_)
		gOrb_->shutdown();
	return TRUE;
}


DDS::DomainId_t DATABASENOTIFICATION_DOMAIN_ID = 1066;
const char* DATABASENOTIFICATION_TYPE = "DatabaseNotification Type";
const char* DATABASENOTIFICATION_TOPIC = "DatabaseNotification Topic";



// set port, transport, keep list of -ORB and -DCPS args to apply later
void ParseArgs(int argc, ACE_TCHAR *argv[],
			   int &port, ACE_TString& transport,
			   std::vector<ACE_TString> &otherTAOArgs,
			   std::vector<ACE_TString> &otherDDSArgs) {
	ACE_Arg_Shifter arg_shifter(argc, argv);
	while (arg_shifter.is_anything_left()) {
		const ACE_TCHAR *currentArg = 0;
		if ((currentArg = arg_shifter.get_the_parameter(ACE_TEXT("-p"))) != 0) {
			port = ACE_OS::atoi(currentArg);
			arg_shifter.consume_arg();
		}
		else if ((currentArg = arg_shifter.get_the_parameter(ACE_TEXT("-t"))) != 0) {
			transport = currentArg;
			arg_shifter.consume_arg();
		}
		else if (arg_shifter.cur_arg_strncasecmp(ACE_TEXT("-ORB")) != -1) {
			// add both the argument itself and its parameter
			otherTAOArgs.push_back(arg_shifter.get_current());
			arg_shifter.consume_arg();
 			otherTAOArgs.push_back(arg_shifter.get_current());
			arg_shifter.consume_arg();
       }
		else if 
			(arg_shifter.cur_arg_strncasecmp(ACE_TEXT("-DCPS")) != -1) {
			// add both the argument itself and its parameter
			otherDDSArgs.push_back(arg_shifter.get_current());
			arg_shifter.consume_arg();
 			otherDDSArgs.push_back(arg_shifter.get_current());
			arg_shifter.consume_arg();
       }		
		else
			arg_shifter.ignore_arg();
	}
}


void BuildCommandLines(int dcpsInfoRepoPort, 
					   ACE_TCHAR *argv0,
					   const std::vector<ACE_TString> &otherTAOArgs,
					   const std::vector<ACE_TString> &otherDDSArgs,
					   ACE_ARGV_T<ACE_TCHAR> &irArgs, 
					   ACE_ARGV_T<ACE_TCHAR> &taoArgs) {

	int dataServerPort = dcpsInfoRepoPort+1;
	ACE_TCHAR dcpsInfoRepoPortBuf[20];
	ACE_OS::itoa(dcpsInfoRepoPort, dcpsInfoRepoPortBuf, 10);
	ACE_TCHAR dataServerPortBuf[20];
	ACE_OS::itoa(dataServerPort, dataServerPortBuf, 10);

	// DCPSInfoRepo -ORBSvcConf lib_tcp.conf -ORBListenEndpoints iiop://:12345 -ORBDottedDecimalAddresses 0
	irArgs.add(argv0);
	irArgs.add(ACE_TEXT("-ORBSvcConf"));
	irArgs.add(ACE_TEXT("lib_tcp.conf"));
	irArgs.add(ACE_TEXT("-ORBListenEndpoints"));
	// for simplicity, no hostname or IP address is specified in order to listen
    // on all interfaces
	irArgs.add(ACE_OS::strdup((ACE_TString(ACE_TEXT("iiop://:")) + dcpsInfoRepoPortBuf).c_str()));
	irArgs.add(ACE_TEXT("-ORBDottedDecimalAddresses"));
	irArgs.add(ACE_TEXT("0"));
	// pass all args to OpenDDS
	for (std::vector<ACE_TString>::const_iterator it = otherDDSArgs.begin(); it!=otherDDSArgs.end(); it++)
		irArgs.add(it->c_str());
	for (std::vector<ACE_TString>::const_iterator it = otherTAOArgs.begin(); it!=otherTAOArgs.end(); it++)
		irArgs.add(it->c_str());

	taoArgs.add(argv0);
	taoArgs.add(ACE_TEXT("-ORBListenEndpoints"));
	taoArgs.add(ACE_OS::strdup((ACE_TString(ACE_TEXT("iiop://:")) + dataServerPortBuf).c_str()));
	taoArgs.add(ACE_TEXT("-ORBSvcConf"));
	taoArgs.add(ACE_TEXT("lib_tcp.conf"));
	taoArgs.add(ACE_TEXT("-ORBDottedDecimalAddresses"));
	taoArgs.add(ACE_TEXT("0"));

	// do not pass DDS args to TAO - just other TAO args
	for (std::vector<ACE_TString>::const_iterator it = otherTAOArgs.begin(); it!=otherTAOArgs.end(); it++)
		taoArgs.add(it->c_str());
}



int ACE_TMAIN(int argc, ACE_TCHAR *argv[]) {
	try {
		// create proper command lines
		int port = 12345;
		ACE_TString transport(ACE_TEXT("SimpleTcp"));
		std::vector<ACE_TString> otherTAOArgs, otherDDSArgs;
		ParseArgs(argc, argv, port, transport, otherTAOArgs, otherDDSArgs);

		ACE_ARGV_T<ACE_TCHAR> irArgs, taoArgs;
		BuildCommandLines(port, argv[0], otherTAOArgs, otherDDSArgs, irArgs, taoArgs);

		// *** FOR DCPSInfoRepo colocation ***
		InfoRepoTask irTask(irArgs);


		DDS::DomainParticipantFactory_var dpf = DDS::DomainParticipantFactory::_nil();
		DDS::DomainParticipant_var participant = DDS::DomainParticipant::_nil();

		try {
			// to allow the server to be shut down gracefully with ctrl-c
			SetConsoleCtrlHandler(ControlCHandler, TRUE);

			// ensure the DCPSInfoRepo has been started before using OpenDDS
			irTask.WaitForStart();

			// *** FOR OpenDDS *** 

			// the DCPSInfoRepo already calls TheParticipantFactoryWithArgs, so just get the factory itself now
			dpf = TheParticipantFactory;

			// create the participant
			participant = dpf->create_participant(
				DATABASENOTIFICATION_DOMAIN_ID,
				PARTICIPANT_QOS_DEFAULT,
				DDS::DomainParticipantListener::_nil(),
				::OpenDDS::DCPS::DEFAULT_STATUS_MASK);
			if (CORBA::is_nil(participant.in())) 
				throw std::exception("create_participant failed");

			// initialize the transport - without a DCPSConfigFile, the transport must be set explicitly
			const OpenDDS::DCPS::TransportIdType TRANSPORT_IMPL_ID = 1;

			// set this via (optional) command-line argument, defaulting to "SimpleTcp"
			TheTransportFactory->get_or_create_configuration(
				TRANSPORT_IMPL_ID, transport.c_str());

			OpenDDS::DCPS::TransportImpl_rch trans_impl =
				TheTransportFactory->create_transport_impl(TRANSPORT_IMPL_ID, 
				OpenDDS::DCPS::AUTO_CONFIG);

			// create the publisher
			DDS::Publisher_var pub = participant->create_publisher(PUBLISHER_QOS_DEFAULT,
				DDS::PublisherListener::_nil(),
				::OpenDDS::DCPS::DEFAULT_STATUS_MASK);
			if (CORBA::is_nil(pub.in())) 
				throw std::exception("create_publisher failed");

			// attach the publisher to the transport
			OpenDDS::DCPS::AttachStatus status = trans_impl->attach(pub.in());
			if (status != OpenDDS::DCPS::ATTACH_OK) {
				std::string msg("Cannot attach to the transport: ");
				switch (status) {
					case OpenDDS::DCPS::ATTACH_BAD_TRANSPORT:
						throw std::exception((msg+"ATTACH_BAD_TRANSPORT").c_str());
					case OpenDDS::DCPS::ATTACH_ERROR:
						throw std::exception((msg+"ATTACH_ERROR").c_str());
					case OpenDDS::DCPS::ATTACH_INCOMPATIBLE_QOS:
						throw std::exception((msg+"ATTACH_INCOMPATIBLE_QOS").c_str());
					default:
						throw std::exception((msg+"unknown status").c_str());
				}
			}

			// register the type
			DatabaseNotificationTypeSupport_var databaseNotification_servant 
				= new DatabaseNotificationTypeSupportImpl();
			if (DDS::RETCODE_OK != databaseNotification_servant->register_type(participant.in(), 
				DATABASENOTIFICATION_TYPE))
				throw std::exception("register_type failed");

			// get the default topic QOS
			DDS::TopicQos default_topic_qos;
			participant->get_default_topic_qos(default_topic_qos);

			// create the topic
			DDS::Topic_var databaseNotification_topic =
			participant->create_topic(DATABASENOTIFICATION_TOPIC, DATABASENOTIFICATION_TYPE,
				default_topic_qos, DDS::TopicListener::_nil(),
				::OpenDDS::DCPS::DEFAULT_STATUS_MASK);
			if (CORBA::is_nil(databaseNotification_topic.in())) 
				throw std::exception("create_topic failed");

			// get the default data writer QOS
			DDS::DataWriterQos dw_default_qos;
			pub->get_default_datawriter_qos(dw_default_qos);

			// create the data writer
			DDS::DataWriter_var databaseNotification_base_dw =
				pub->create_datawriter(databaseNotification_topic.in(),
				dw_default_qos,
				DDS::DataWriterListener::_nil(),
				::OpenDDS::DCPS::DEFAULT_STATUS_MASK);
			if (CORBA::is_nil(databaseNotification_base_dw.in()))
				throw std::exception("create_datawriter failed");

			DatabaseNotificationDataWriter_var databaseNotification_dw
				= DatabaseNotificationDataWriter::_narrow(databaseNotification_base_dw.in());
			if (CORBA::is_nil(databaseNotification_dw.in())) 
				throw std::exception("DatabaseNotificationDataWriter could not be narrowed");


			// *** FOR TAO ***

			// initialize the ORB
			int taoArgc = taoArgs.argc();
			ACE_TCHAR **taoArgv = taoArgs.argv();
			// use a named ORB so won't share with DCPSInfoRepo unnamed ORB
			CORBA::ORB_var orb = CORBA::ORB_init(taoArgc, taoArgv, "DataServer"); 
			gOrb_ = orb.in();

			// get a reference to the RootPOA
			CORBA::Object_var obj = orb->resolve_initial_references("RootPOA");
			PortableServer::POA_var poa = PortableServer::POA::_narrow(obj.in());

			// activate the POAManager
			PortableServer::POAManager_var mgr = poa->the_POAManager();
			mgr->activate();

			// open the database - DataServer (for now) maintains one open database connection for all clients
			DataLib::Database database;
			if (!database.Open())
				throw std::exception("cannot open the database");

			// create the database servant
			Database_i servant(%database, databaseNotification_dw.in());
			PortableServer::ObjectId_var oid = poa->activate_object(&servant);
			CORBA::Object_var database_obj = poa->id_to_reference(oid.in());

			// for corbaloc - start server as -ORBListenEndpoints iiop://machine:port
			CORBA::String_var ior_str = orb->object_to_string(database_obj.in());
			CORBA::Object_var tobj = orb->resolve_initial_references("IORTable");
			IORTable::Table_var table = IORTable::Table::_narrow(tobj.in());
			table->bind("DataServer", ior_str.in());
			std::cout << "DataServer bound to IORTable" << std::endl;

			// accept requests from clients
			orb->run();
			orb->destroy();
		}
		catch (CORBA::Exception& ex) {
			std::cerr << "CORBA exception: " << ex << std::endl;
		}
		catch (std::exception& ex) {
			std::cerr << "Exception: " << ex.what() << std::endl;
		}

		// DDS cleanup is performed by the shutdown of the DCPSInfoRepo

	}
	catch (CORBA::Exception& ex) {
		std::cerr << "CORBA exception: " << ex << std::endl;
	}
	catch (std::exception& ex) {
		std::cerr << "Exception: " << ex.what() << std::endl;
	}

	return 0;
}