package com.ociweb.sett;

/**
 * An interface for defining celestial bodies (start, planets, moons, etc.).
 */
public interface CelestialBody {
    /**
     * The Gravitational Constant in units of m/(kgs).
     */
    public static final double GRAVITATIONAL_CONSTANT = 6.67300e-11;

    Properties getProperties();

    interface Properties {
        /**
         * @return the mass of this celestial body, in kg
         */
        double getMass();

        /**
         * @return the radius of this celestial body, in m.
         */
        double getRadius();

        /**
         * @return an array containing the satellites which orbit this
         * celestial body, or null if this celestial body has no satellites
         */
        Satellite[] getSatellites();

        /**
         * Returns the name of this celestial body. Implementation of this
         * method can be provided by {@link Enum}.
         *
         * @return the name of this celestial body
         */
        String name();

        /**
         * Prints a summary for the celestial body and any satellites it has.
         */
        void printSummary();
    }

    /**
     * An immutable representation of the properties of the {@link
     * CelestialBody}. Immutability is necessary to honor the contract for
     * enums.
     */
    static final class ImmutableProperties implements Properties {
        private final double mass;
        private final double radius;
        private final String name;
        private final Satellite[] satellites;

        public ImmutableProperties(String name, double mass, double radius,
                Satellite... satellites) {
            this.name = name;
            this.mass = mass;
            this.radius = radius;
            this.satellites = satellites == null ? null : satellites.clone();
        }

        @Override
        public double getMass() {
            return mass;
        }

        @Override
        public double getRadius() {
            return radius;
        }

        @Override
        public Satellite[] getSatellites() {
            return satellites == null ? null : satellites.clone();
        }

        @Override
        public String name() {
            return name;
        }

        public void printSummary() {
            System.out.println("Summary for " + name() + ":");
            System.out.printf("Surface Gravity: %1$e m/s\n\n",
                    Util.calcSurfaceGravity(this));

            if (satellites != null) {
                for (Satellite satellite : getSatellites()) {
                    satellite.getProperties().printSummary();
                }
            }
        }
    }

    /**
     * A static class that is used to simulate the logic that would be
     * implemented in an abstract superclass for Satellite enums, if Java
     * allowed such a thing.
     */
    static class Util {

        // private constructor to ensure static-only access
        private Util() {
        }

        /**
         * Calculates and returns the surface gravity of the given {@link
         * CelestialBody}. The result is given in m/s.
         *
         * @param celestialBody the body whose surface gravity is being
         *                      calculated
         * @return the surface gravity, in m/s
         */
        public static double calcSurfaceGravity(Properties celestialBody) {
            return GRAVITATIONAL_CONSTANT * celestialBody.getMass() / Math.pow(
                    celestialBody.getRadius(), 2.0);
        }
    }
}
