package com.ociweb.sett.initial;

/**
 * A common abstract superclass for celestial bodies.
 */
public class CelestialBody {
    /**
     * The Gravitational Constant, used for calculations: 6.67300e-1
     * m/(kgs).
     */
    public static final double GRAVITATIONAL_CONSTANT = 6.67300e-11;

    private final String name;
    private final double mass; // mass in kg
    private final double radius; // radius in m
    private Satellite[] satellites;

    protected CelestialBody(String name, double mass, double radius) {
        this.name = name;
        this.mass = mass;
        this.radius = radius;
    }

    /**
     * @return the surface gravity, in m/s
     */
    public double getSurfaceGravity() {
        return GRAVITATIONAL_CONSTANT * mass / Math.pow(radius, 2.0);
    }

    /**
     * @return the mass of this celestial body, in kg
     */
    public final double getMass() {
        return mass;
    }

    /**
     * @param satellites an array containing the satellites which orbit this
     *                   celestial body, or null if this celestial body has no
     *                   satellites
     */
    public void setSatellites(Satellite[] satellites) {
        this.satellites = satellites;
    }

    /**
     * @return an array containing the satellites which orbit this celestial
     *         body, or null if this celestial body has no satellites
     */
    public final Satellite[] getSatellites() {
        return satellites;
    }

    /**
     * @return the name of this celestial body
     */
    public final String getName() {
        return name;
    }
}
