package com.ociweb.sett.initial;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * A class representing a generic satellite (planet, moon, etc.).
 */
public final class Satellite extends CelestialBody {
    private final CelestialBody parentBody;
    private final double semiMajorAxis; // in m

    public Satellite(String name, CelestialBody parentBody, double mass,
            double radius, double semiMajorAxis) {
        super(name, mass, radius);
        this.parentBody = parentBody;
        this.semiMajorAxis = semiMajorAxis;
    }

    /**
     * @return the celestial body around which this celestial body orbits
     */
    public CelestialBody getParentBody() {
        return parentBody;
    }

    /**
     * @return the orbital period, in seconds
     */
    public double getOrbitalPeriod() {
        return 2 * Math.PI * Math.sqrt(Math.pow(semiMajorAxis, 3.0) /
                (GRAVITATIONAL_CONSTANT * parentBody.getMass()));
    }

    /**
     * Prints a summary of the calculated orbital and physical characteristics
     * of this class and any of its own satellites.
     */
    public void printSummary() {
        System.out.println("Summary for " + getParentage());

        long orbitalPeriod = (long) getOrbitalPeriod();
        System.out.printf("Orbital Period: %1$d hours/%2$d days\n",
                TimeUnit.SECONDS.toHours(orbitalPeriod),
                TimeUnit.SECONDS.toDays(orbitalPeriod));

        System.out.printf("Surface Gravity: %1$e m/s\n\n",
                getSurfaceGravity());

        Satellite[] satellites = getSatellites();
        if (satellites != null) {
            for (Satellite child : satellites) {
                child.printSummary();
            }
        }
    }

    /**
     * Generates and returns a String listing all the orbital parents of this
     * body. So, for the Earth's moon, it would return ", satellite of EARTH,
     * satellite of SOL".
     *
     * @return a String showing orbital parentage
     */
    private String getParentage() {
        List<CelestialBody> parents = new ArrayList<CelestialBody>();
        CelestialBody parentBody = this.parentBody;
        parents.add(parentBody);
        while (parentBody instanceof Satellite) {
            Satellite satelliteParent = (Satellite) parentBody;
            parentBody = satelliteParent.getParentBody();
            parents.add(parentBody);
        }

        StringBuilder sb = new StringBuilder(getName());
        for (CelestialBody parent : parents) {
            sb.append(", satellite of ");
            sb.append(parent.getName());
        }
        sb.append(":");

        return sb.toString();
    }

}
