import QtQuick 2.6
import QtQuick.Window 2.2
import QtQuick.Controls 2.1
import QtQuick.Layouts 1.3

Window {
    visible: true
    width: 640
    height: 480
    title: qsTr("Calculator")

    property double currentTotal: 0
    property bool startNumber: true
    property string displayField: "0"
    property string previousOp: "="
    property string error: qsTr("ERROR")

    function getTotal() {
        return currentTotal;
    }

    function setTotal(total) {
        currentTotal = parseFloat(total);
    }

    function add(n) {
        currentTotal += parseFloat(n);
    }

    function subtract(n) {
        currentTotal -= parseFloat(n);
    }

    function multiply(n) {
        currentTotal *= parseFloat(n);
    }

    function divide(n) {
        // check for zero
        currentTotal /= parseFloat(n);
    }

    function clear() {
        startNumber = true;
        displayField = "0";
        previousOp = "=";
        setTotal("0");
    }

    function clearEntry() {
        startNumber = true;
        displayField = "0";
    }

    function backspace() {
        if (displayField.length > 0)
            displayField = displayField.slice(0, displayField.length-1)
    }

    function negate() {
        if (displayField !== error) {
            if (displayField.length > 0) {
                if (displayField[0] === '-')
                    displayField = displayField.slice(1, displayField.length);
                else
                    displayField = "-" + displayField;
                startNumber = true;
            }
        }
    }

    function applyOp(op) {
        if (startNumber) {
            clear();
            displayField = error;  // expecting a number
        } else {
            startNumber = true;
            if (previousOp === "=")
                setTotal(displayField);
            else if (previousOp === "+")
                add(displayField);
            else if (previousOp === "-")
                subtract(displayField);
            else if (previousOp === "*")
                multiply(displayField);
            else if (previousOp === "/")
                divide(displayField);

            displayField = getTotal().toString();
            previousOp = op;
        }
    }

    function applyDigit(digit) {
        if (startNumber) {
            // first digit - clear field and set
            displayField = digit;
            startNumber = false;
        } else
            // append digit
            displayField = displayField + digit;
    }


    ColumnLayout {
        anchors.fill: parent

        TextEdit {
            id: textEdit
            text: displayField
            verticalAlignment: Text.AlignVCenter
            horizontalAlignment: Text.AlignRight
            anchors.left: parent.left
            anchors.right: parent.right

            color: "#000000"
            font.family: "Abel"
            font.pointSize: 16
            anchors.margins: 10

            Rectangle {
                anchors.fill: parent
                color: "transparent"
                border.width: 1
            }
        }


        GridLayout {
            id: grid
            rows: 4
            columns: 4
            anchors.left: parent.left
            anchors.right: parent.right

            GridButton {
                text: "CE"
                onClicked: clearEntry()
            }

            GridButton {
                text: "C"
                onClicked: clear()
            }
            GridButton {
                text: "\u2190" // "<-"
                onClicked: backspace()
            }
            GridButton {
                text: "/"
                onClicked: applyOp("/")
            }

            GridButton {
                text: "7"
                onClicked: applyDigit("7")
            }
            GridButton {
                text: "8"
                onClicked: applyDigit("8")
            }
            GridButton {
                text: "9"
                onClicked: applyDigit("9")
            }
            GridButton {
                text: "*"
                onClicked: applyOp("*")
            }

            GridButton {
                text: "4"
                onClicked: applyDigit("4")
            }
            GridButton {
                text: "5"
                onClicked: applyDigit("5")
            }
            GridButton {
                text: "6"
                onClicked: applyDigit("6")
            }
            GridButton {
                text: "-"
                onClicked: applyOp("-")
            }

            GridButton {
                text: "1"
                onClicked: applyDigit("1")
            }
            GridButton {
                text: "2"
                onClicked: applyDigit("2")
            }
            GridButton {
                text: "3"
                onClicked: applyDigit("3")
            }
            GridButton {
                text: "+"
                onClicked: applyOp("+")
            }

            GridButton {
                text: "\u00B1" // "+/-"
                onClicked: negate()
            }

            GridButton {
                text: "0"
                onClicked: applyDigit("0")
            }
            GridButton {
                text: "."
                onClicked: applyDigit(".")
            }
            GridButton {
                text: "="
                onClicked: applyOp("=")
            }

        }

    }
}
