// Copyright (c) 2010, Object Computing, Inc.
// All rights reserved.

#ifdef _MSC_VER
# pragma warning(disable:4996) // Disable VC warning from Boost serialization
# pragma warning(disable:4099) // Disable VC warning from Boost serialization
#endif

#include "zmq.hpp"
#include <iostream>
#include <sstream>
#include <boost/archive/binary_oarchive.hpp>
#include <boost/archive/binary_iarchive.hpp>
#include <boost/iostreams/device/array.hpp>
#include <boost/iostreams/stream_buffer.hpp>
#include <boost/ref.hpp>
#include <Functions_T.h>
#include <MarketData.h>
#include <QuoteRequest.h>


int main (int argc, char *argv [])
{
  try {
    if (argc < 4) {
      std::cout << "usage: ZeromqTypedPublisher <roundtrip-count>  <bind-to-sub> [<connect-to-pub>]+\n"
                << "e.g.:  ZeromqTypedPublisher  10000 tcp://eth0:54321 tcp://spider:54322 tcp://spider:54323\n"
                << "          use a literal IP address and port for each endpoint\n"
                << std::endl;
    }
    size_t roundtrip_count = atoi (argv [1]);
    const char* bind_to_sub = argv[2];

    zmq::context_t ctx (1, 1);
    zmq::socket_t pub(ctx,ZMQ_PUB);
    zmq::socket_t sub(ctx,ZMQ_SUB);
    sub.setsockopt(ZMQ_SUBSCRIBE,"MarketData\x00",11);
    sub.setsockopt(ZMQ_SUBSCRIBE,"QuoteRequest\x00",13);
    sub.bind(bind_to_sub);

    size_t num_subscribers = 0;
    for (int argi = 3; argi < argc; ++argi) {
      pub.connect(argv[argi]);
      ++num_subscribers;
    }

    printf("Entering send loop -- %d messages\n", roundtrip_count);

    MiddlewareNewsBrief::MarketData md = 
      MiddlewareNewsBrief::MarketData::createTestData();
    MiddlewareNewsBrief::QuoteRequest qr = 
      MiddlewareNewsBrief::QuoteRequest::createTestData();

    void* watch = zmq_stopwatch_start ();

    for (size_t i = 0; i < roundtrip_count; i++) 
    {
      std::ostringstream request(std::ios::binary);
      boost::archive::binary_oarchive oa(request);

      // Send 10 MarketDatas for each QuoteRequest
      std::string topic_name;
      if (i % 10 == 5) 
      {
        topic_name = MiddlewareNewsBrief::QuoteRequest::TOPIC;
        MiddlewareNewsBrief::stream(qr,oa,i);
      }
      else 
      {
        topic_name = MiddlewareNewsBrief::MarketData::TOPIC;
        MiddlewareNewsBrief::stream(md,oa,i);
      }

      size_t request_length = request.str().length() + topic_name.length() + 1;
      zmq::message_t msg(request_length);
      memset(msg.data(),0,msg.size());
      memcpy(msg.data(),topic_name.c_str(),topic_name.length());
      memcpy(reinterpret_cast<char*>(msg.data()) + topic_name.length() + 1,request.str().data(),request.str().length());

      pub.send(msg);

      for (size_t jj = 0; jj < num_subscribers; ++jj) 
      {
        // Wait for echoed message from each subscriber
        zmq::message_t reply_msg;
        sub.recv(&reply_msg);

        char* reply_buffer = reinterpret_cast<char*>(reply_msg.data());
        
        // "topic" name should be first N characters, null-terminated
        std::string reply_topic_name(reply_buffer);

        std::istringstream reply_stream(
          std::string(reply_buffer + reply_topic_name.length() + 1,
                      reply_msg.size() - (reply_topic_name.length() + 1)),
          std::ios::binary);
        boost::archive::binary_iarchive ia(reply_stream);

        if (reply_topic_name == MiddlewareNewsBrief::MarketData::TOPIC) 
        {
          MiddlewareNewsBrief::MarketData mdReply;
          if (MiddlewareNewsBrief::check(mdReply,ia,i) != 0) { return -1; }
        } 
        else if (reply_topic_name == MiddlewareNewsBrief::QuoteRequest::TOPIC)
        {
          MiddlewareNewsBrief::QuoteRequest qrReply;
          if (MiddlewareNewsBrief::check(qrReply,ia,i) != 0) { return -1; }
        }
        else 
        {
          std::cerr << "Received invalid topic name: " << reply_topic_name.c_str() << std::endl;
          return -1;
        }
      }
    }

    unsigned long elapsed = zmq_stopwatch_stop (watch);
    double latency = (double) elapsed / (roundtrip_count * 2.0) / (double)num_subscribers;

    printf ("roundtrip count: %d\n", (int) roundtrip_count);
    printf ("\n\naverage latency: %.3f [us]\n\n\n", (double) latency);
  
    return 0;
  } catch (std::exception &e) {
    std::cout << "An error occurred: " << e.what() << std::endl;
    return 1;
  }
}
