﻿// Copyright (c) 2010, Object Computing, Inc.
// All rights reserved.

using System;
using System.Collections.Generic;
using System.Text;
using System.Runtime.InteropServices;
using System.Diagnostics;
using System.Runtime.Serialization;
using System.Runtime.Serialization.Formatters.Binary;
using System.IO;
using MiddlewareNewsBrief;

class ZmqTypedPublisher
{

   static unsafe int Main(string[] args)
   {
      const String QUOTE_TOPIC = "QuoteRequest";
      const String MARKET_DATA_TOPIC = "MarketData";

      if (args.Length < 3)
      {
         Console.Out.WriteLine("usage: ZmqTypedPublisher <roundtrip-count> " +
              "<bind-to-sub> [<connect-to-pub>]+\n");
         Console.Out.WriteLine(" e.g.: ZmqTypedPublisher 10000 tcp://10.201.200.72:54321 tcp://10.201.200.72:54322\n");
         return 1;
      }

      int roundtripCount = Convert.ToInt32(args[0]);
      String bind_to_sub = args[1];

      //  Initialise 0MQ infrastructure
      ZMQ.Context ctx = new ZMQ.Context(1, 1, 0);
      ZMQ.Socket pub = ctx.Socket(ZMQ.PUB);
      ZMQ.Socket sub = ctx.Socket(ZMQ.SUB);
      sub.SetSockOpt(ZMQ.SUBSCRIBE, MARKET_DATA_TOPIC);
      sub.SetSockOpt(ZMQ.SUBSCRIBE, QUOTE_TOPIC);

      Console.Out.WriteLine("Binding to " + bind_to_sub);
      sub.Bind(bind_to_sub);

      int num_subscribers = 0;
      for (int i = 2; i < args.Length; ++i)
      {
         Console.Out.WriteLine("Connecting to " + args[i]);
         pub.Connect(args[i]);
         ++num_subscribers;
      }

      // Create two messages to send, and stream each message
      // to a byte array
      IFormatter formatter = new BinaryFormatter();
      ASCIIEncoding encoding = new System.Text.ASCIIEncoding();

      MarketData md = MarketData.createTestData();
      QuoteRequest qr = QuoteRequest.createTestData();

      Console.Out.WriteLine("Sending messages -- " + num_subscribers + " subscribers, "
         + roundtripCount + " messages");

      //  Start measuring the time.
      System.Diagnostics.Stopwatch watch;
      watch = new Stopwatch();
      watch.Start();

      //  Start sending messages.
      for (uint i = 0; i < roundtripCount; i++)
      {
         // Send 90% MarketData messages
         if (i % 10 == 5)
         {
            qr.isEcho = false;
            qr.counter = i;
            byte[] quoteMsg = serialize(qr, QUOTE_TOPIC, formatter, encoding);
            pub.Send(quoteMsg);
         }
         else
         {
            md.isEcho = false;
            md.counter = i;
            byte[] mdMsg = serialize(md, MARKET_DATA_TOPIC, formatter, encoding);
            pub.Send(mdMsg);
         }

         byte[] echoMsg;
         for (int jj = 0; jj < num_subscribers; ++jj)
         {
            sub.Recv(out echoMsg);

            // Get the "Topic" from the front of the byte array
            int topicEndIndex = Array.IndexOf(echoMsg, (byte)'\x00');
            String topic = new String(encoding.GetChars(echoMsg, 0, topicEndIndex));

            // Deserialize the echo, which should be the same message
            object echo = deserialize(echoMsg, topic, formatter);

            if (topic.Equals(MARKET_DATA_TOPIC))
            {
               MarketData mdEcho = (MarketData)echo;
               Debug.Assert(mdEcho.isEcho == true, "Subscriber forgot to set isEcho flag to true");
               Debug.Assert(mdEcho.counter == i, "Counter mismatch in subscriber's reply");
            }
            else if (topic.Equals(QUOTE_TOPIC))
            {
               QuoteRequest qrEcho = (QuoteRequest)echo;
               Debug.Assert(qrEcho.isEcho == true, "Subscriber forgot to set isEcho flag to true");
               Debug.Assert(qrEcho.counter == i, "Counter mismatch in subscriber's reply");
            }
            else
            {
               Console.Out.WriteLine("ERROR: received topic " + topic);
               return -1;
            }
         }
      }

      //  Stop measuring the time.
      watch.Stop();
      Int64 elapsedTime = watch.ElapsedTicks;

      //  Print out the test parameters.
      Console.Out.WriteLine("roundtrip count: " + roundtripCount);

      //  Compute and print out the latency.
      double latency = (double)(elapsedTime) / roundtripCount / 2 *
          1000000 / Stopwatch.Frequency / (double)num_subscribers;
      Console.Out.WriteLine("\n\nYour average latency is {0} [us]\n\n",
          latency.ToString("f2"));

      return 0;
   }

    static byte[] serialize(object obj, String topic, IFormatter formatter, ASCIIEncoding encoding)
    {
       MemoryStream stream = new MemoryStream();

       // "topic" for ZeroMQ
       stream.Write(encoding.GetBytes(topic + '\x00'),
                    0,
                    topic.Length + 1);

       formatter.Serialize(stream, obj);
       stream.Close();
       return stream.ToArray();
    }

    static object deserialize(byte[] msg, String topic, IFormatter formatter)
    {
       MemoryStream stream = new MemoryStream(msg);

       // Seek past "topic" for ZeroMQ
       stream.Seek(topic.Length + 1, SeekOrigin.Begin);
       object obj = formatter.Deserialize(stream);
       stream.Close();
       return obj;
    }
}
