// Copyright (c) 2010, Object Computing, Inc.
// All rights reserved.

#include "DDSUtil.h"
#include <string>
#include <sstream>
#include <stdexcept>

// OpenDDS
#include <dds/DCPS/Service_Participant.h>
#include <dds/DCPS/SubscriberImpl.h>
#include <dds/DCPS/PublisherImpl.h>
#include <dds/DCPS/Marked_Default_Qos.h>
#include <dds/DCPS/transport/framework/TheTransportFactory.h>

namespace MiddlewareNewsBrief {

DDS::Subscriber_ptr 
DDSUtil::create_subscriber(DDS::DomainParticipant_ptr participant,
                           const int transport_impl_id,
                           const char* partition)
{
  DDS::SubscriberQos sub_qos;
  if (DDS::RETCODE_OK != participant->get_default_subscriber_qos(sub_qos))
  {
    std::ostringstream msg;
    msg << "get_default_subscriber_qos failed." << std::ends;
    throw std::runtime_error(msg.str());
  }
  if (partition != 0) {
    sub_qos.partition.name.length(1);
    sub_qos.partition.name[0] = partition;
  }

  DDS::Subscriber_var subscriber = participant->create_subscriber(sub_qos,
                                                                  0,
                                                                  0);
  if (subscriber == 0) 
  {
    std::ostringstream msg;
    msg << "create_subscriber failed." << std::ends;
    throw std::runtime_error(msg.str());
  }

  // Initialize the transport; the TRANSPORT_IMPL_ID must match the
  // value in the configuration file.
  OpenDDS::DCPS::TransportImpl_rch transport_impl =
    TheTransportFactory->create_transport_impl (transport_impl_id, 
                                                OpenDDS::DCPS::AUTO_CONFIG);

  // Attach the subscriber to the transport.
  OpenDDS::DCPS::SubscriberImpl* sub_impl =
    dynamic_cast<OpenDDS::DCPS::SubscriberImpl*>(subscriber.in());
  if (0 == sub_impl) 
  {
    std::ostringstream msg;
    msg << "Failed to obtain subscriber servant" << std::ends;
    throw std::runtime_error(msg.str());
  }

  OpenDDS::DCPS::AttachStatus status = 
    sub_impl->attach_transport(transport_impl.in());
  if (status != OpenDDS::DCPS::ATTACH_OK) 
  {
    std::ostringstream msg;
    msg << "Failed to attach to the subscriber transport. Status = "
        << status << std::ends;
    throw std::runtime_error(msg.str());
  }

  return subscriber._retn();
}

DDS::Publisher_ptr 
DDSUtil::create_publisher(DDS::DomainParticipant_ptr participant,
                          const int transport_impl_id,
                          const char* partition)
{
  DDS::PublisherQos pub_qos;
  if (DDS::RETCODE_OK != participant->get_default_publisher_qos(pub_qos))
  {
    std::ostringstream msg;
    msg << "get_default_publisher_qos failed." << std::ends;
    throw std::runtime_error(msg.str());
  }
  if (partition != 0) {
    pub_qos.partition.name.length(1);
    pub_qos.partition.name[0] = partition;
  }

  DDS::Publisher_var publisher = participant->create_publisher(pub_qos,
                                                               0,
                                                               0);
  if (publisher == 0) 
  {
    std::ostringstream msg;
    msg << "create_publisher failed." << std::ends;
    throw std::runtime_error(msg.str());
  }

  // Initialize the transport; the TRANSPORT_IMPL_ID must match the
  // value in the configuration file.
  OpenDDS::DCPS::TransportImpl_rch transport_impl =
    TheTransportFactory->create_transport_impl (transport_impl_id, 
                                                OpenDDS::DCPS::AUTO_CONFIG);

  // Attach the publisher to the transport.
  OpenDDS::DCPS::PublisherImpl* pub_impl =
    dynamic_cast<OpenDDS::DCPS::PublisherImpl*>(publisher.in());
  if (0 == pub_impl) 
  {
    std::ostringstream msg;
    msg << "Failed to obtain publisher servant" << std::ends;
    throw std::runtime_error(msg.str());
  }
  OpenDDS::DCPS::AttachStatus status = 
    pub_impl->attach_transport(transport_impl.in());
  if (status != OpenDDS::DCPS::ATTACH_OK) 
  {
    std::ostringstream msg;
    msg << "Failed to attach to the subscriber transport. Status = "
        << status << std::ends;
    throw std::runtime_error(msg.str());
  }
  
  return publisher._retn();
}


DDS::DataReader_ptr
DDSUtil::create_datareader(DDS::DomainParticipant_ptr participant,
                           DDS::Subscriber_ptr subscriber,
                           std::string topic_name,
                           const std::string& type_name,
                           DDS::DataReaderListener_ptr listener)
{
  if (topic_name.empty())
  {
    topic_name = type_name;
  }

  DDS::Topic_var the_topic = 
    DDS::Topic::_narrow(participant->lookup_topicdescription(topic_name.c_str()));

  if (the_topic == 0) {
    DDS::TopicQos topic_qos;
    participant->get_default_topic_qos(topic_qos);

    the_topic = participant->create_topic( topic_name.c_str(),
                                           type_name.c_str(),
                                           topic_qos,
                                           0,
                                           0);
    if (the_topic == 0)
    {
      std::ostringstream msg;
      msg  << "create_topic for " << topic_name.c_str() 
           << " failed." << std::ends;
      throw std::runtime_error(msg.str());
    }
  }

  // Get QoS to use for our DataReader
  DDS::DataReaderQos dr_qos;
  subscriber->get_default_datareader_qos (dr_qos);
  dr_qos.reliability.kind = DDS::RELIABLE_RELIABILITY_QOS;
  dr_qos.history.kind = DDS::KEEP_ALL_HISTORY_QOS;
        
  // Create a DataWriter for the Telemetry topic, and make sure it is non-nil
  DDS::DataReader_var reader =
    subscriber->create_datareader(the_topic,
                                  dr_qos,
                                  listener,
                                  DDS::DATA_AVAILABLE_STATUS);
  if (reader == 0) 
  {
    std::ostringstream msg;
    msg << "create_reader for " 
        << topic_name.c_str() << " failed." << std::ends;
    throw std::runtime_error(msg.str());
  }
  
  return reader._retn();
}


DDS::DataWriter_ptr
DDSUtil::create_datawriter(DDS::DomainParticipant_ptr participant,
                           DDS::Publisher_ptr publisher,
                           std::string topic_name,
                           const std::string& type_name)
{
  if (topic_name.empty())
  {
    topic_name = type_name;
  }

  // Create a topic for the type...
  DDS::Topic_var the_topic = 
    DDS::Topic::_narrow(participant->lookup_topicdescription(topic_name.c_str()));

  if (the_topic == 0) {
    DDS::TopicQos topic_qos;
    participant->get_default_topic_qos(topic_qos);

    the_topic = participant->create_topic( topic_name.c_str(),
                                           type_name.c_str(),
                                           topic_qos,
                                           0,
                                           0);
    if (the_topic == 0)
    {
      std::ostringstream msg;
      msg  << "create_topic for " << topic_name.c_str() 
           << " failed." << std::ends;
      throw std::runtime_error(msg.str());
    }
  }

  // Get QoS to use for our DataWriter
  DDS::DataWriterQos dw_qos;
  publisher->get_default_datawriter_qos (dw_qos);
  dw_qos.reliability.kind = DDS::RELIABLE_RELIABILITY_QOS;
  dw_qos.history.kind = DDS::KEEP_ALL_HISTORY_QOS;
  dw_qos.writer_data_lifecycle.autodispose_unregistered_instances = false;
  
  // Create a DataWriter for the Telemetry topic, and make sure it is non-nil
  DDS::DataWriter_var dw =
    publisher->create_datawriter(the_topic,
                                 dw_qos,
                                 0,
                                 0);
  if (dw == 0) 
  {
    std::ostringstream msg;
    msg << "create_datawriter for " 
        << topic_name.c_str() << " failed." << std::ends;
    throw std::runtime_error(msg.str());
  }

  return dw._retn();
}



} // MiddlewareNewsBrief
