// Copyright (c) 2010, Object Computing, Inc.
// All rights reserved.

#ifdef _MSC_VER
# pragma once
#endif
#ifndef MIDDLEWARENEWSBRIEF_DDSUTIL_ECHO_RECEIVER_T_H
#define MIDDLEWARENEWSBRIEF_DDSUTIL_ECHO_RECEIVER_T_H

#include "DDSUtil.h"
#include "Profiler.h"

// OpenDDS
#include <dds/DCPS/Service_Participant.h>
#include <dds/DCPS/SubscriberImpl.h>
#include <ace/Synch.h>

#include <string>

namespace MiddlewareNewsBrief {

    template <typename DDS_STRUCT_T> struct DDSTypeDetails;

    template <typename DDS_STRUCT_T>
    class EchoReceiver
      : public virtual OpenDDS::DCPS::LocalObject<DDS::DataReaderListener>
    {
    public:
      typedef DDSTypeDetails<DDS_STRUCT_T> Details;

      EchoReceiver();
      virtual ~EchoReceiver();

      void init(DDS::DomainParticipant_ptr participant,
                DDS::Subscriber_ptr subscriber,
                DDS::DataWriter_ptr source,
                CORBA::ULong DDS_STRUCT_T::* timestampField = 0);

      void recv_echo();
      
      virtual void on_data_available(
        DDS::DataReader_ptr reader
      )
      throw (CORBA::SystemException);

      virtual void on_requested_deadline_missed (
        DDS::DataReader_ptr reader,
        const DDS::RequestedDeadlineMissedStatus & status)
        throw (CORBA::SystemException)
      {}

      virtual void on_requested_incompatible_qos (
        DDS::DataReader_ptr reader,
        const DDS::RequestedIncompatibleQosStatus & status)
      throw (CORBA::SystemException)
      {}

      virtual void on_liveliness_changed (
        DDS::DataReader_ptr reader,
        const DDS::LivelinessChangedStatus & status)
      throw (CORBA::SystemException)
      {}

      virtual void on_subscription_matched (
        DDS::DataReader_ptr reader,
        const DDS::SubscriptionMatchedStatus & status
      )
      throw (CORBA::SystemException)
      {}

      virtual void on_sample_rejected(
        DDS::DataReader_ptr reader,
        const DDS::SampleRejectedStatus& status
      )
      throw (CORBA::SystemException)
      {}

      virtual void on_sample_lost(
        DDS::DataReader_ptr reader,
        const DDS::SampleLostStatus& status
      )
      throw (CORBA::SystemException){}
    
    private:
      const std::string on_data_available_;
      CORBA::ULong DDS_STRUCT_T::* timestampField_;

      ACE_Mutex lock_;
      ACE_Condition<ACE_Mutex> condition_;
      size_t echo_count_;
    };



template <typename DDS_STRUCT_T>
EchoReceiver<DDS_STRUCT_T>::EchoReceiver()
  : on_data_available_("Echo<" + Details::type_name + ">::on_data_availa()")
  , timestampField_(0)
  , condition_(lock_)
  , echo_count_(0)
{
}

template <typename DDS_STRUCT_T>
EchoReceiver<DDS_STRUCT_T>::~EchoReceiver()
{
}

template <typename DDS_STRUCT_T>
void
EchoReceiver<DDS_STRUCT_T>::init(
  DDS::DomainParticipant_ptr participant,
  DDS::Subscriber_ptr subscriber,
  DDS::DataWriter_ptr source,
  CORBA::ULong DDS_STRUCT_T::* timestampField)
{
  this->timestampField_ = timestampField;
  
  DDS::Topic_var the_topic = source->get_topic();
  CORBA::String_var topic_name = the_topic->get_name();

  DDS::DataReader_var dr =
    DDSUtil::create_datareader<typename Details::TypeSupportImpl>(participant,
                                                                  subscriber,
                                                                  topic_name.in(),
                                                                  this);
}

template <typename DDS_STRUCT_T>
void
EchoReceiver<DDS_STRUCT_T>::recv_echo()
{
  ACE_Guard<ACE_Mutex> guard(this->lock_);
  while (this->echo_count_ == 0) {
    this->condition_.wait();
  }
  --this->echo_count_;
}


template <typename DDS_STRUCT_T>
void
EchoReceiver<DDS_STRUCT_T>::on_data_available(DDS::DataReader_ptr reader)
  throw (CORBA::SystemException)
{
  typename Details::Reader::_var_type dr = 
    Details::Reader::_narrow(reader);
  if (dr == 0) 
  {
    std::cerr << "EchoReceiver<DDS_STRUCT_T>::on_data_available: _narrow failed." 
              << std::endl;
    return;
  }

  // Take a sample

  DDS_STRUCT_T sample;
  DDS::SampleInfo si ;
  DDS::ReturnCode_t status = dr->take_next_sample(sample, si);

  while (status == DDS::RETCODE_OK) 
  {
    if (si.instance_state == DDS::ALIVE_INSTANCE_STATE
        && (this->timestampField_ == 0 || sample.*(this->timestampField_) > 0)
        )
    {
      // Profile
      if (this->timestampField_ != 0) {
        MIDDLEWARENEWSBRIEF_PROFILE_POINT_W_START(this->on_data_available_.c_str(),sample.*(this->timestampField_)); 
      }

      ACE_Guard<ACE_Mutex> guard(this->lock_);
      ++this->echo_count_;
      this->condition_.broadcast();
    }

    // See if there's another one
    status = dr->take_next_sample(sample, si);
  } 
}

} // 

#endif 

