﻿// Copyright (c) 2010, Object Computing, Inc.
// All rights reserved.

using System;
using System.Collections.Generic;
using System.Text;
using System.Runtime.InteropServices;
using System.Diagnostics;
using System.Runtime.Serialization;
using System.Runtime.Serialization.Formatters.Binary;
using System.IO;
using System.Threading;
using MiddlewareNewsBrief;

public class BoostTypedDotnetPublisher
{
   BoostPubProxy proxy_;
   uint echo_count_ = 0;
   uint message_count_ = 0;

   IFormatter formatter_ = new BinaryFormatter();
   ASCIIEncoding encoding_ = new System.Text.ASCIIEncoding();

   const String QUOTE_TOPIC = "QuoteRequest";
   const String MARKET_DATA_TOPIC = "MarketData";

   static unsafe int Main(string[] args)
   {
      BoostTypedDotnetPublisher me = new BoostTypedDotnetPublisher();
      me.Run(args);
      return 0;
   }

   public BoostTypedDotnetPublisher()
   {
   }

   int Run(string[] args)
   {
      if (args.Length < 3)
      {
         Console.Out.WriteLine("usage: BoostTypedDotnetPublisher <roundtrip-count> " +
              "[<subscriber host> <subscriber port>]+\n");
         Console.Out.WriteLine(" e.g.: BoostTypedDotnetPublisher 10000 localhost 54321 localhost 54322\n");
         return 1;
      }

      // Initialize OpenDDS; publisher transport Id is 1 for publisher process
      this.proxy_ = new BoostPubProxy();
      this.proxy_.ProcessNotification += new BoostPubProxy.EventHandler(OnEchoReceived);

      // Create two messages to send, and stream each message
      // to a byte array

      MarketData md = MarketData.createTestData();
      QuoteRequest qr = QuoteRequest.createTestData();

      Console.Out.WriteLine("Sending messages -- " + this.proxy_.get_num_subscribers() 
         + " subscribers, " + this.proxy_.get_num_messages() + " messages");

      uint roundtripCount = this.proxy_.get_num_messages();
      uint num_subscribers = this.proxy_.get_num_subscribers();

      //  Start measuring the time.
      System.Diagnostics.Stopwatch watch;
      watch = new Stopwatch();
      watch.Start();

      //  Start sending messages.
      for (uint i = 0; i < roundtripCount; i++)
      {
         this.message_count_ = i;
         
         // Send 90% MarketData messages
         if (i % 10 == 5)
         {
            qr.isEcho = false;
            qr.counter = i;
            byte[] quoteMsg = serialize(qr, QUOTE_TOPIC, this.formatter_, this.encoding_);
            this.proxy_.write(quoteMsg);
         }
         else
         {
            md.isEcho = false;
            md.counter = i;
            byte[] mdMsg = serialize(md, MARKET_DATA_TOPIC, this.formatter_, this.encoding_);
            this.proxy_.write(mdMsg);
         }

         // Wait for echo from each subscriber
         for (int jj = 0; jj < num_subscribers; ++jj)
         {
            this.TakeEcho();
         }
      }

      //  Stop measuring the time.
      watch.Stop();
      Int64 elapsedTime = watch.ElapsedTicks;

      //  Print out the test parameters.
      Console.Out.WriteLine("roundtrip count: " + roundtripCount);

      //  Compute and print out the latency.
      double latency = (double)(elapsedTime) / roundtripCount / 2 *
          1000000 / Stopwatch.Frequency / (double)(num_subscribers);
      Console.Out.WriteLine("\n\nYour average latency is {0} [us]\n\n",
          latency.ToString("f2"));

      return 0;
   }

   // invoked when a sample arrives
   void OnEchoReceived(object parent, byte[] msg)
   {
      // deserialize the buffer into a Quote or MarketData object

      // Get the "Topic" from the front of the byte array
      int topicEndIndex = Array.IndexOf(msg, (byte)'\x00');
      String topic = new String(encoding_.GetChars(msg, 0, topicEndIndex));

      object obj = deserialize(msg, topic, this.formatter_);

      if (topic.Equals(MARKET_DATA_TOPIC))
      {
         MarketData md = (MarketData)obj;
         Debug.Assert(md.isEcho == true, "Subscriber forgot to set isEcho flag to true");
         Debug.Assert(md.counter == this.message_count_, "Counter mismatch in subscriber's reply");
      }
      else if (topic.Equals(QUOTE_TOPIC))
      {
         QuoteRequest qr = (QuoteRequest)obj;
         Debug.Assert(qr.isEcho == true, "Subscriber forgot to set isEcho flag to true");
         Debug.Assert(qr.counter == this.message_count_, "Counter mismatch in subscriber's reply");
      }
      else
      {
         Console.Out.WriteLine("ERROR: received topic " + topic);
         return;
      }

      lock (this)
      {
         ++this.echo_count_;
         Monitor.PulseAll(this);
      }
   }

   void TakeEcho()
   {
      lock (this)
      {
         while (this.echo_count_ == 0) {
            Monitor.Wait(this);
         }
         --this.echo_count_;
      }
    }

    static byte[] serialize(object obj, String topic, IFormatter formatter, ASCIIEncoding encoding)
    {
       MemoryStream stream = new MemoryStream();

       // "topic" for ZeroMQ
       stream.Write(encoding.GetBytes(topic + '\x00'),
                    0,
                    topic.Length + 1);

       formatter.Serialize(stream, obj);
       stream.Close();
       return stream.ToArray();
    }

    static object deserialize(byte[] msg, String topic, IFormatter formatter)
    {
       MemoryStream stream = new MemoryStream(msg);

       // Seek past "topic" for ZeroMQ
       stream.Seek(topic.Length + 1, SeekOrigin.Begin);
       object obj = formatter.Deserialize(stream);
       stream.Close();
       return obj;
    }
}
