// Copyright (c) 2010, Object Computing, Inc.
// All rights reserved.

#ifdef _MSC_VER
# pragma warning(disable:4005) // Disable VC warning about duplicate macros;
                               // Boost and ACE define same macros for Windows
#endif

#include "MarketDataTypeDetails.h"
#include <MarketDataTypeSupportImpl.h>

#include "DdsUtilPch.h"
#include "DDSUtil.h"
#include "DataWriter_T.h"
#include "DoneTokenManager.h"

// OpenDDS
#include <dds/DCPS/Service_Participant.h>
#include <dds/DCPS/Marked_Default_Qos.h>
#include <dds/DCPS/PublisherImpl.h>
#include <dds/DCPS/transport/framework/TheTransportFactory.h>

#include <iostream>
#include <set>

// Profiler
#include "Profiler.h"

void usage() {
  std::cout 
    << "-num numMessages   : Number of messages to write\n"
    << "-ns numNubscribers : number of subscribers\n"
    << std::endl;
}

MiddlewareNewsBrief::MarketData* createMarketData();
MiddlewareNewsBrief::MarketDataEntry* createMarketDataEntry();
MiddlewareNewsBrief::QuoteRequest* createQuoteRequest();
MiddlewareNewsBrief::RelatedSym* createRelatedSym();


MiddlewareNewsBrief::MarketData* createMarketData()
{
  MiddlewareNewsBrief::MarketData_var data = new MiddlewareNewsBrief::MarketData();

  data->applVersionID = "1.0";
  data->messageType = "100";
  data->senderCompID = "Test Exchange";
  data->msgSeqNum = 4;
  data->sendingTime = 00162635;
  data->tradeDate = 20100422;

  MiddlewareNewsBrief::MarketDataEntry_var entry = createMarketDataEntry();
  data->mdEntries.length(3);
  data->mdEntries[0] = entry;
  data->mdEntries[1] = entry;
  data->mdEntries[2] = entry;

  return data._retn();
}

MiddlewareNewsBrief::MarketDataEntry* createMarketDataEntry()
{
   MiddlewareNewsBrief::MarketDataEntry_var entry = new MiddlewareNewsBrief::MarketDataEntry();

   entry->mdUpdateAction = 1;
   entry->mdPriceLevel = 2;
   entry->mdEntryType = "7";
   entry->openCloseSettleFlag = 3;
   entry->securityID = 99;
   entry->securityIDSource = 9;
   entry->rptSeq = 2;
   entry->mdEntryPx = 100.0;
   entry->mdEntryTime = 12345;
   entry->mdEntrySize = 50;
   entry->numberOfOrders = 10;
   entry->tradingSessionID = "2";
   entry->netChgPrevDay = 10.0;
   entry->tradeVolume = 30;
   entry->tradeCondition = "W";
   entry->tickDirection = "0";
   entry->quoteCondition = "C";
   entry->aggressorSide = 2;
   entry->matchEventIndicator = "1";

   return entry._retn();
}


MiddlewareNewsBrief::QuoteRequest* createQuoteRequest()
{
   MiddlewareNewsBrief::QuoteRequest_var req = new MiddlewareNewsBrief::QuoteRequest();

   req->applVersionID = "1.0";
   req->messageType = "100";
   req->senderCompID = "Test Exchange";
   req->msgSeqNum = 4;
   req->sendingTime = 00162635;
   req->quoteReqID = "R";

   MiddlewareNewsBrief::RelatedSym_var sym = createRelatedSym();
   req->related.length(3);
   req->related[0] = sym;
   req->related[1] = sym;
   req->related[2] = sym;

   return req._retn();
}


MiddlewareNewsBrief::RelatedSym* createRelatedSym()
{
   MiddlewareNewsBrief::RelatedSym_var sym = new MiddlewareNewsBrief::RelatedSym();

   sym->symbol = "[N/A]";
   sym->orderQuantity = 25;
   sym->side = 1;
   sym->transactTime = 00162635;
   sym->quoteType = 1;
   sym->securityID = 99;
   sym->securityIDSource = 9;

   return sym._retn();
}




int main(int argc, char** argv)
{
  const int DOMAIN_ID = 8675309;

  size_t numMessages = 10000;
  size_t numSubscribers = 1;

  // Command-Line Arguments
  //
  //    -num numMessages : Number of messages to write
  //    -ns numNubscribers : number of subscribers
  //
  int i = 0;
  while (i < argc) {
    if (std::string(argv[i]) == "-num" && (i+1) < argc) {
      numMessages = boost::lexical_cast<size_t>(argv[++i]);
    } else if (std::string(argv[i]) == "-ns" && (i+1) < argc) {
       numSubscribers = boost::lexical_cast<size_t>(argv[++i]);
    } else if (std::string(argv[i]) == "-?" ) {
      usage();
      return 0;
    }
    ++i;
  }

  std::cout << "Number of messages is " << numMessages << std::endl;

  try
  {
    //
    // OpenDDS Init
    //

    // Initialize, and create a DomainParticipantFactory
    DDS::DomainParticipantFactory_var factory = 
      TheParticipantFactoryWithArgs(argc, argv);

    //  Create the DomainParticipant
    DDS::DomainParticipant_var participant = 
      factory->create_participant(DOMAIN_ID,
                                  PARTICIPANT_QOS_DEFAULT,
                                  0,
                                  0);
    if (participant == 0) 
    {
      std::cerr << "create_participant failed." << std::endl;
      return -1;
    }

    // Create a publisher for the topics
    const int TRANSPORT_IMPL_ID = 1;
    DDS::Publisher_var publisher = 
      MiddlewareNewsBrief::DDSUtil::create_publisher(participant.in(),
                                                     TRANSPORT_IMPL_ID,
                                                     "Primary");

    // Create a subscriber for the Echo of the two topics
    const int TRANSPORT_IMPL_ID_2 = 2;
    DDS::Subscriber_var subscriber = 
      MiddlewareNewsBrief::DDSUtil::create_subscriber(participant.in(),
                                                      TRANSPORT_IMPL_ID_2,
                                                      "Secondary");

    // Initialize the DoneToken manager, which publishes a "done" token
    MiddlewareNewsBrief::DoneTokenManager doneToken;
    doneToken.initWriter(participant,publisher,subscriber);

    //
    // DataWriters, one per message type
    //

    typedef MiddlewareNewsBrief::MarketData MarketData;
    boost::shared_ptr<MiddlewareNewsBrief::DataWriter<MarketData> > 
      mdDataWriter(new MiddlewareNewsBrief::DataWriter<MarketData>());
    mdDataWriter->init(participant, 
                       publisher, 
                       subscriber);
    mdDataWriter->sleepUsecBetweenWrites(0);

    typedef MiddlewareNewsBrief::QuoteRequest QuoteRequest;
    boost::shared_ptr<MiddlewareNewsBrief::DataWriter<QuoteRequest> > 
      qrDataWriter(new MiddlewareNewsBrief::DataWriter<QuoteRequest>());
    qrDataWriter->init(participant, 
                        publisher, 
                        subscriber);
    qrDataWriter->sleepUsecBetweenWrites(0);

    boost::this_thread::sleep(boost::posix_time::seconds(3));

    MiddlewareNewsBrief::MarketData_var md = createMarketData();
    MiddlewareNewsBrief::QuoteRequest_var qr = createQuoteRequest();

    std::cout << "Writing..." << std::endl;

    MIDDLEWARENEWSBRIEF_PROFILER_TIME_TYPE start = 
      MIDDLEWARENEWSBRIEF_PROFILER_GET_TIME;

    for (size_t i = 0; i < numMessages; ++i) 
    {
      if (i % 5)
      {
        qr->is_echo = false;
        qr->counter = i;

        qrDataWriter->onMessage(qr
  #ifdef MIDDLEWARENEWSBRIEF_PROFILER_ENABLE
                                ,0
  #endif
                                );
        for (size_t jj = 0; jj < numSubscribers; ++jj) {      
          qrDataWriter->recv_echo();
        }
      }
      else
      {
        md->is_echo = false;
        md->counter = i;

        mdDataWriter->onMessage(md
  #ifdef MIDDLEWARENEWSBRIEF_PROFILER_ENABLE
                                ,0
  #endif
                                );
        for (size_t jj = 0; jj < numSubscribers; ++jj) {      
          mdDataWriter->recv_echo();
        }
      }
    }

    MIDDLEWARENEWSBRIEF_PROFILER_TIME_TYPE end = 
      MIDDLEWARENEWSBRIEF_PROFILER_GET_TIME;

    double latency = (double)(end-start) / ((double)numMessages * 2.0)  / (double)numSubscribers;
    std::cout << "\n\nAverage latency in " 
              << MIDDLEWARENEWSBRIEF_PROFILER_TIME_UNITS
              << ": " << latency << "\n\n" << std::endl;

    doneToken.writeDone("done");

    // Give the samples time to make it to the subscribers
    DDS::Duration_t duration;
    duration.sec = DDS::DURATION_INFINITE_SEC;
    duration.nanosec = DDS::DURATION_INFINITE_NSEC;
    publisher->wait_for_acknowledgments(duration);

    // OpenDDS Cleanup
    doneToken.fini(publisher.in());
    if (publisher != 0) 
    {
      publisher->delete_contained_entities();
      participant->delete_publisher(publisher.in());
      publisher = DDS::Publisher::_nil();
    }
    if (participant != 0) 
    {
      participant->delete_contained_entities();
    }
    if (factory != 0) 
    {
      factory->delete_participant(participant.in ());
    }
    TheTransportFactory->release();
    TheServiceParticipant->shutdown();  
  }
  catch (std::exception & e)
  {
    std::cerr << "Exception: " << e.what() << std::endl;
    return -1;
  }
  catch (CORBA::Exception& e) 
  {
    std::cerr << "Exception: " << e << std::endl;
    return -1;
  }
  return 0;
}

