// Copyright (c) 2010, Object Computing, Inc.
// All rights reserved.

#ifdef _MSC_VER
# pragma warning(disable:4996) // Disable VC warning from Boost serialization
# pragma warning(disable:4099) // Disable VC warning from Boost serialization
#endif

#include "zmq.hpp"
#include "MarketData.pb.h"
#include <iostream>
#include <cstdlib>
#include <iostream>
#include <boost/detail/atomic_count.hpp>

const std::string MARKET_DATA_TOPIC = "MarketData";
const std::string QUOTE_REQUEST_TOPIC = "QuoteRequest";

int main (int argc, char *argv [])
{
  try {
    if (argc != 4) {
      std::cout << "usage: ZeromqTypedProtobufSubscriber <roundtrip-count> <bind-to-sub> <connect-to-pub>\n"
                << "e.g.:  ZeromqTypedProtobufSubscriber 10000 tcp://eth0:54322 tcp://spider:54321\n"
                << "          use a literal IP address and port for each endpoint\n"
                << std::endl;
        return 1;
    }
    size_t roundtrip_count = atoi (argv [1]);
    const char* bind_to_sub = argv [2];
    const char* connect_to_pub = argv [3];

    zmq::context_t ctx (1, 1);
    zmq::socket_t pub (ctx, ZMQ_PUB);
    pub.connect(connect_to_pub);

    zmq::socket_t sub (ctx, ZMQ_SUB);
    sub.setsockopt(ZMQ_SUBSCRIBE,"MarketData\x00",11);
    sub.setsockopt(ZMQ_SUBSCRIBE,"QuoteRequest\x00",13);
    sub.bind(bind_to_sub);

    boost::detail::atomic_count msg_counter(0);

    printf("Entering recv loop -- %d messages\n", 
           roundtrip_count);

    pbMarketData md;
    pbQuoteRequest qr;
    for (size_t i = 0; i < roundtrip_count; i++) {
      zmq::message_t msg;
      sub.recv (&msg);

      char* msg_buffer = reinterpret_cast<char*>(msg.data());
      
      // "topic" name should be first N characters, null-terminated
      std::string topic_name(msg_buffer);
      google::protobuf::Message* payload;

      if (topic_name == MARKET_DATA_TOPIC) 
      {
        md.ParseFromArray(msg_buffer + topic_name.length() + 1,
                          msg.size() - (topic_name.length() + 1));

        if (md.is_echo() == true || md.counter() != i)
        {
          std::cerr << "MarketData reply isEcho or counter mismatch" << std::endl;
        }
        md.set_is_echo(true);
        payload = &md;
      } 
      else if (topic_name == QUOTE_REQUEST_TOPIC)
      {
        qr.ParseFromArray(msg_buffer + topic_name.length() + 1,
                          msg.size() - (topic_name.length() + 1));

        if (qr.is_echo() == true || qr.counter() != i)
        {
          std::cerr << "QuoteRequest reply isEcho or counter mismatch" << std::endl;
        }
        qr.set_is_echo(true);
        payload = &qr;
      }
      else 
      {
        std::cerr << "Received invalid topic name: " << topic_name.c_str() << std::endl;
        return -1;
      }

      // Echo it back
      size_t reply_length = payload->ByteSize() + topic_name.length() + 1;
      zmq::message_t reply_msg(reply_length);
      memset(reply_msg.data(),0,reply_msg.size());
      memcpy(reply_msg.data(),topic_name.c_str(),topic_name.length());
      payload->SerializeToArray(reinterpret_cast<char*>(reply_msg.data())+topic_name.length()+1,payload->ByteSize());

      pub.send(reply_msg,0);

      ++msg_counter;
    }
    printf("Finished receiving messages\n");
    
    return 0;

  } catch (std::exception &e) {
    std::cout << "An error occurred: " << e.what() << std::endl;
    return 1;
  }
}
