// Copyright (c) 2010, Object Computing, Inc.
// All rights reserved.

#include "zmq.hpp"
#include <iostream>


int main (int argc, char *argv [])
{
  try {
    if (argc < 5) {
      std::cout << "usage: ZeromqPublisher <message-size> <roundtrip-count>  <bind-to-sub> [<connect-to-pub>]+\n"
                << "e.g.:  ZeromqPublisher 1000 10000 tcp://eth0:54321 tcp://spider:54322 tcp://spider:54323\n"
                << "          use a literal IP address and port for each endpoint\n"
                << std::endl;
    }
    size_t message_size = atoi (argv [1]);
    size_t roundtrip_count = atoi (argv [2]);
    const char* bind_to_sub = argv[3];

    zmq::context_t ctx (1, 1);
    zmq::socket_t pub(ctx,ZMQ_PUB);
    zmq::socket_t sub(ctx,ZMQ_SUB);
    sub.setsockopt(ZMQ_SUBSCRIBE,"DataFeed\x00",9);
    sub.bind(bind_to_sub);

    size_t num_subscribers = 0;
    for (int argi = 4; argi < argc; ++argi) {
      pub.connect(argv[argi]);
      ++num_subscribers;
    }

    zmq::message_t msg(message_size);
    memset(msg.data(), 0, message_size);
    memcpy(msg.data(), "DataFeed\x00", 9);

    printf("Entering send loop -- %d messages, size = %d\n", 
           roundtrip_count,
           message_size);

    void* watch = zmq_stopwatch_start ();

    for (size_t i = 0; i != roundtrip_count; i++) {
      pub.send(msg);  
      for (size_t jj = 0; jj < num_subscribers; ++jj) {
        // Wait for echoed message from each subscriber
        sub.recv(&msg);
        if (msg.size() != message_size) {
          std::cout << "Message of incorrect size received: " << msg.size() 
                    << std::endl;
          return -1;
        }
      }
    }

    unsigned long elapsed = zmq_stopwatch_stop (watch);
    double latency = (double) elapsed / (roundtrip_count * 2.0) / (double)num_subscribers;

    printf ("message size: %d [B]\n", (int) message_size);
    printf ("roundtrip count: %d\n", (int) roundtrip_count);
    printf ("\n\naverage latency: %.3f [us]\n\n\n", (double) latency);
  
    return 0;
  } catch (std::exception &e) {
    std::cout << "An error occurred: " << e.what() << std::endl;
    return 1;
  }
}
